﻿/*!
\file rdb5api.js
	
	新版的支持订阅和异步请求的实时库api

\remark 
	依赖 md5.js，适合rdb2025.3（inver 5122）及以后版本	，浏览器支持 ES6 async/await
\author
	蒋勇 eMail: kipway@outlook.com  本文件代码可任意使用和修改无需通知我
\update 
	2025-4-28 v2.0.0alpha
*/
class Rdb5Api {
	constructor() {
		//实时库连接部分
		this._wsurl = null;
		this._wsusr = null;   //用户名
		this._wspsw = null;   //密码
		this._wsprotocol = "rdb5"; // Sec-WebSocket-Protocol 值，应用层定义,默认"rdb5"
		this._wsobject = null;  // WS连接对象
		this._loginst = false;  //当前登录状态

		this._seqno = 0; // 全局连续递增的序列号，仅供nextseqno使用，不要直接使用。
		this._version = "";
		this._build = "";
		this._timeoutcall = 20000;//调用超时毫秒数, 默认20秒

		this._ssid_wsurl = "_rdb5_wsurl";//默认的sessionStorage存储url的键名
		this._ssid_wsuser = "_rdb5_wsuser";//默认的sessionStorage存储账号的键名
		this._ssid_wspswd = "_rdb5_wspswd";//默认的sessionStorage存储密码的键名

		this._OnRecvMessage = (event) => { }; //接收到的原始消息(event)，调试用
		this._OnLoginStatus = (bool) => { }; //登录状态改变 (bool), 应用层安装此函数当登录成功后订阅快照或者SOE
		this._OnRecvObject = (jsoncls) => { }; //处理已解析的json对象 (jsoncls)，处理接收到的消息。
		this._OnWsClosed = (event) => { }; // 已关闭 (event)
				
		this._mapaWait = new Map();//异步回调，seqno为key，Promise为值
		this.timer_checkcalltimeout = 0; //检查调用超时的定时器
		this._CheckCallTimeOut = () => {
			let curtimestamp = Date.now();//检查await超时
			let vdel = [];
			for (let [key, obj] of this._mapaWait) {
				if (Math.abs(curtimestamp - obj.timestamp) >= this._timeoutcall) {
					obj.reject(new Error("call timeout."));
					vdel.push(key);
				}
			}
			for (const seqno of vdel) {
				this._mapaWait.delete(seqno);
			}
		};
	}

	/**
	 * 递增序列号
	 * @returns 返回1-2147483646序列号
	 */
	nextseqno() {
		this._seqno++;
		if (this._seqno === 2147483647)
			this._seqno = 1;
		return this._seqno;
	}

	/**
	 * 创建websocket连接，并安装消息处理函数
	 * @param {string} wsurl 连接地址 ws://192.168.1.59:921
	 * @param {string} usr 登录用户名
	 * @param {string} psw 登录密码
	 * @returns 返回bool值 true or false
	 */
	ws_connect(wsurl, usr, psw) {
		this._loginst = false;
		this._wsurl = wsurl;
		this._wsusr = usr;
		this._wspsw = psw;
		try {
			this._wsobject = new WebSocket(this._wsurl, this._wsprotocol);
		} catch (e) {
			console.log("new WebSocket Err:", e);
			return false;
		}
		//安装连接成功处理
		this._wsobject.onopen = (event) => {
			let oreq = {};
			oreq.request = "rdb_login";
			oreq.seqno = this.nextseqno();
			oreq.name = this._wsusr;
			oreq.timefmt = "iso";
			this.ws_send(JSON.stringify(oreq));
		};
		//安装接收WS消息处理
		this._wsobject.onmessage = (event) => {
			this._OnRecvMessage(event);
			let cls = null;
			try {
				cls = JSON.parse(event.data);
			}
			catch (e) {
				console.log("onmessage JSON.parse Err:", e);
				return;
			}
			if (cls.response === "rdb_login") {
				if (cls.status !== 0) { //登录失败关闭连接
					console.log("onlogin Err:", cls);
					this._OnLoginStatus(false);
					this._wsobject.close();
					this._wsobject = null;
					return;
				}
				let oreq = {};
				oreq.request = "rdb_auth";
				oreq.seqno = this.nextseqno();
				oreq.vals = hex_md5(cls.vals + hex_md5(this._wspsw).toUpperCase()).toUpperCase();
				this.ws_send(JSON.stringify(oreq));
			} else if (cls.response === "rdb_auth") {
				if (cls.status !== 0) { //验证失败关闭连接
					console.log("onauth Err:", cls);
					this._OnLoginStatus(false);
					this._wsobject.close();
					this._wsobject = null;
				} else {
					console.log("login success!", cls);
					sessionStorage.setItem(this._ssid_wsurl, this._wsurl);
					sessionStorage.setItem(this._ssid_wsuser, this._wsusr);
					sessionStorage.setItem(this._ssid_wspswd, this._wspsw);
					if (cls.version != undefined) {
						this._version = cls.version;
					}
					if (cls.build != undefined) {
						this._build = cls.build;
					}
					this._OnLoginStatus(true);
					this._loginst = true;
					if (!this.timer_checkcalltimeout) {
						this.timer_checkcalltimeout = setInterval(this._CheckCallTimeOut, 1000);
					}
				}
			}
			else {
				let executor = this._mapaWait.get(cls.seqno);
				if (undefined !== executor) { //先处理异步请求
					executor.resolve(cls);
					this._mapaWait.delete(cls.seqno);
				}
				else {
					this._OnRecvObject(cls); //其他消息，由应用层处理
				}
			}
		};
		//安装关闭处理
		this._wsobject.onclose = (event) => {
			console.log("ws_onclose", event);
			this._loginst = false;
			for (let [key, obj] of this._mapaWait) {
				obj.reject("websocket closed."); //先逐个reject后再全部清空
			}
			this._mapaWait.clear();
			this._OnWsClosed(event);
		};
		return true;
	}

	/**
	* 获取当前登录状态
	* @returns 返回bool值 true or false
	*/
	ws_islogin() {
		return this._loginst;
	}

	/**
	 * 自动登录,从sessionStorage中读取登录参数登录，用于页面切换后自动登录。
	 * @returns true or false
	 */
	ws_autologin() //自动登录
	{
		if (!this._wsobject) {
			try {
				let wsurl = sessionStorage.getItem(this._ssid_wsurl);
				let usr = sessionStorage.getItem(this._ssid_wsuser);
				let psw = sessionStorage.getItem(this._ssid_wspswd);
				if (!wsurl || !usr || !psw)
					return false;
				return this.ws_connect(wsurl, usr, psw);
			} catch (e) {
				console.log("ws_connect Err:", e);
				return false;
			}
		}
		return true;
	}

	/**
	 * 主动断开，从sessionStorage去除登录参数登录，用于页面切换后重新登录
	 * @returns 无
	 */
	ws_disconnet()//主动断开
	{
		try {
			sessionStorage.removeItem(this._ssid_wsurl);
			sessionStorage.removeItem(this._ssid_wsuser);
			sessionStorage.removeItem(this._ssid_wspswd);
			if (this._wsobject) {
				this._wsobject.close();
				this._wsobject = null;
			}
			this._loginst = false;
		}
		catch (e) {
			console.log("ws_disconnet Err:", e);
		}
	}

	/**
	 * 发送消息
	 * @param {string} jstr 发送的JSON字符串
	 * @returns true or false
	 */
	ws_send(jstr)//发送消息
	{
		try {
			if (this._wsobject === null || this._wsobject.readyState !== this._wsobject.OPEN)
				return false;
			this._wsobject.send(jstr);
		} catch (e) {
			return false;
		}
		return true;
	}

	/**
	 * 从标签名中取出子站名，即':'前的字符串
	 * @param {any} tagname
	 * @returns 返回string或者null
	 */
	static tagStation(tagname) {
		var i;
		for (i = 0; i < tagname.length; i++) {
			if (tagname.charAt(i) === ':') {
				return tagname.slice(0, i);
			}
			else if (tagname.charAt(i) === '.')
				break;
		}
		return null;
	}

	/**
	 * 异步请求消息。
	 * @param {object} oreq 请求对象，请按照《rdb_data_exchange_protocol.pdf》填写，rdb_call会填写seqno
	 * @returns {object} 返回一个Promise对象，状态未知，调用者需await等待完成
	 */
	async rdb_call(oreq) {
		oreq.seqno = this.nextseqno();
		return new Promise((fresolve, freject) => {
			let executor = { resolve: fresolve, reject: freject, timestamp: Date.now() };
			try {
				let jstr = JSON.stringify(oreq);
				this._mapaWait.set(oreq.seqno, executor);
				if (!this.ws_send(jstr)) {
					throw new Error("ws_send error, not login or connect.");
				}
			} catch (e) {
				this._mapaWait.delete(oreq.seqno);
				freject(e);
			}
		});
	}
	

	/**
	* 控制输出
	* @param {string} stag 标签名
	* @param {string} datatype 数据类型,可以是 "digtal", "int32", "float", "double", "int64", "string", "object"之一
	* @param {any} val 标签值
	* @return {boolean}
	*/
	async rdb_ctrlout(stag, datatype, val) //控制输出，返回输出命令seqno
	{
		var oreq = {};
		oreq.request = "rdb_writedevice";
		oreq.vals = {};
		oreq.vals.N = stag;
		oreq.vals.DT = datatype;
		oreq.vals.V = val;
		return this.rdb_call(oreq);
	}	
}



