﻿/*!
@file rdbsvgview.js
	新版的实时库web-view，封装processview生成的web版svg工况运行画面对象。
	包含三大部分：
	1）实时库websocket连接，数据访问异步接口。
	2）svg图形的动作触发处理。
	3）鼠标事件捕获处理。

	pvedit生成页面时较以前版本，为了封装需要做两处改动。
	1) svg图形内部的自动id需要加上svg标签id作为前缀。
	2) 按照新的js脚本用法生成script标签再body的最后。

@remark
	依赖 md5.js，适合rdb2025.3（inver 5122）及以后版本，浏览器支持 ES6 async/await
@author
	蒋勇 eMail: kipway@outlook.com  本文件代码可任意使用和修改无需通知我
@update
	2025-5-26 ver 2.0.1 初始化递归进入foreignObject，主动断开加通知。
	2025-5-7 ver 2.0.0
*/
class RdbView {
	constructor() {
		//实时库连接部分
		this._wsurl = null;
		this._wsusr = null;   //用户名
		this._wspsw = null;   //密码
		this._wsprotocol = "rdb5"; // Sec-WebSocket-Protocol 值，应用层定义,默认"rdb5"
		this._wsobject = null;  // WS连接对象
		this._loginst = false;  //当前登录状态

		this._seqno = 0; // 全局连续递增的序列号，仅供nextseqno使用，不要直接使用。
		this._version = "";
		this._build = "";
		this._timeoutcall = 20000;//调用超时毫秒数, 默认20秒

		this._ssid_wsurl = "_rdbsvg_wsurl";//默认的sessionStorage存储url的键名
		this._ssid_wsuser = "_rdbsvg_wsuser";//默认的sessionStorage存储账号的键名
		this._ssid_wspswd = "_rdbsvg_wspswd";//默认的sessionStorage存储密码的键名

		this._OnRecvMessage = (event) => { }; //接收到的原始消息(event)
		this._OnLoginStatus = (bool) => { }; //登录状态改变 (bool)
		this._OnRecvObject = (jsoncls) => { }; //处理已解析的json对象 (jsoncls)
		this._OnWsClosed = (event) => { }; // 已关闭 (event)

		this._SubscribeTags = () => { //登录成功后订阅标签
			let oreq = {};
			oreq.request = "rdb_sscsnaps";
			oreq.seqno = this.nextseqno();
			oreq.tags = new Array();
			for (const tagname of this._ssctags) {
				oreq.tags.push({ name: tagname, mode: 1 });
			}
			this.ws_send(JSON.stringify(oreq));
		};
		this._snapGetInterval = 1000; //不能订阅的子站标签快照读取间隔毫秒。
		this._seqnoGetTagAttr = -1; //序列号和其他应用读取标签属性分开。
		this._GetTagsAttr = () => { //登录成功后读取使用的标签的属性
			let oreq = {};
			oreq.request = "rdb_tagget";
			oreq.seqno = this.nextseqno();
			oreq.tags = new Array();
			this._seqnoGetTagAttr = oreq.seqno;//保存度属性的seqno
			for (const tagname of this._ssctags) {
				oreq.tags.push(tagname);
			}
			for (const tagname of this._subtags) {
				oreq.tags.push(tagname);
			}
			this.ws_send(JSON.stringify(oreq));
		};

		//svg对象部分
		this._svgid = "svg01"; //svg标签的ID
		this._ssctags = [];//需要订阅的标签表，字符串数组
		this._subtags = [];//子站标签，需要定时读取。
		this._actions = [];//动作表,对象数组

		this._mapLocalVal = new Map(); //本地全局变量
		this._mapSnapshot = new Map(); //实时库快照
		this._mapTagAttr = new Map();//实时库标签属性
		this._mapTrigger = new Map();//触发器map,标签名为key, _actions数组下标为值.例如：{"d0.f01.pv",[2,10]},用于标签快照值中的标签名快速获得this._actions中动作，执行动作
		this._mapPidActionsIdx = new Map();//以pid为key，_actions数组下标为值,例如：{"tagval_10",[1,2]},用于快速获得this._actions中动作，执行鼠标事件
		this._mapaWait = new Map();//异步回调，seqno为key，Promise为值

		this._stationTags = new Map(); //子站标签对象数组,key为子站名，值为rdb_getsnap命令请求对象

		this.timer_checkcalltimeout = 0; //检查调用超时的定时器
		this._CheckCallTimeOut = () => {
			let curtimestamp = Date.now();//检查await超时
			let vdel = [];
			for (let [key, obj] of this._mapaWait) {
				if (Math.abs(curtimestamp - obj.timestamp) >= this._timeoutcall) {
					obj.reject(new Error("call timeout."));
					vdel.push(key);
				}
			}
			for (const seqno of vdel) {
				this._mapaWait.delete(seqno);
			}
		};
		this.timer_getsubsnaps = 0; //读取子站标签快照的定时器
	}

	/**
	 * 递增序列号
	 * @returns 返回1-2147483646序列号
	 */
	nextseqno() {
		this._seqno++;
		if (this._seqno === 2147483647)
			this._seqno = 1;
		return this._seqno;
	}

	/**
	 * 创建websocket连接，并安装消息处理函数
	 * @param {string} wsurl 连接地址 ws://192.168.1.59:921
	 * @param {string} usr 登录用户名
	 * @param {string} psw 登录密码
	 * @returns 返回bool值 true or false
	 */
	ws_connect(wsurl, usr, psw) {
		this._loginst = false;
		this._wsurl = wsurl;
		this._wsusr = usr;
		this._wspsw = psw;
		try {
			this._wsobject = new WebSocket(this._wsurl, this._wsprotocol);
		} catch (e) {
			console.log("new WebSocket Err:", e);
			return false;
		}
		//安装连接成功处理
		this._wsobject.onopen = (event) => {
			let oreq = {};
			oreq.request = "rdb_login";
			oreq.seqno = this.nextseqno();
			oreq.name = this._wsusr;
			oreq.timefmt = "iso";
			this.ws_send(JSON.stringify(oreq));
		};
		//安装接收WS消息处理
		this._wsobject.onmessage = (event) => {
			this._OnRecvMessage(event);
			let cls = null;
			try {
				cls = JSON.parse(event.data);
			}
			catch (e) {
				console.log("onmessage JSON.parse Err:", e);
				return;
			}
			if (cls.response === "rdb_login") {
				if (cls.status !== 0) { //登录失败关闭连接
					console.log("onlogin Err:", cls);
					this._OnLoginStatus(false);
					this._wsobject.close();
					this._wsobject = null;
					return;
				}
				let oreq = {};
				oreq.request = "rdb_auth";
				oreq.seqno = this.nextseqno();
				oreq.vals = hex_md5(cls.vals + hex_md5(this._wspsw).toUpperCase()).toUpperCase();
				this.ws_send(JSON.stringify(oreq));
			} else if (cls.response === "rdb_auth") {
				if (cls.status !== 0) { //验证失败关闭连接
					console.log("onauth Err:", cls);
					this._OnLoginStatus(false);
					this._wsobject.close();
					this._wsobject = null;
				} else {
					console.log("login success!", cls);
					sessionStorage.setItem(this._ssid_wsurl, this._wsurl);
					sessionStorage.setItem(this._ssid_wsuser, this._wsusr);
					sessionStorage.setItem(this._ssid_wspswd, this._wspsw);
					if (cls.version != undefined) {
						this._version = cls.version;
					}
					if (cls.build != undefined) {
						this._build = cls.build;
					}
					this._OnLoginStatus(true);
					this._loginst = true;
					this._GetTagsAttr();//先读取标签属性
					this._SubscribeTags();//再订阅标签
					this._subStationGetSnaps();//登录成功后先读一遍子站标签快照

					if (!this.timer_checkcalltimeout) {
						this.timer_checkcalltimeout = setInterval(this._CheckCallTimeOut, 1000);
					}
					if (!this.timer_getsubsnaps && document.getElementById(this._svgid)) {
						this.timer_getsubsnaps = setInterval(this._subStationGetSnaps, this._snapGetInterval);
					}
				}
			}
			else {
				let executor = this._mapaWait.get(cls.seqno);
				if (executor) { //先处理异步请求
					executor.resolve(cls);
					this._mapaWait.delete(cls.seqno);
				}
				else {
					if (cls.seqno === this._seqnoGetTagAttr && cls.response === "rdb_tagget") {
						let uptime = Date.now(); //更新时间。
						for (let tagobj of cls.vals) {
							tagobj.updateTime = uptime; //用于控制是不频繁从实时库读取
							this._mapTagAttr.set(tagobj.name, tagobj);
							this._SetActionDesc(tagobj);
						}
						this._seqnoGetTagAttr = -1;
					}
					else if (cls.response === "rdb_pushsnaps") {
						for (let tagv of cls.vals) {
							tagv.isSubscript = 1;//订阅的,用于使用时判断是否需要从实时库读取。
							this._mapSnapshot.set(tagv.N, tagv);
							this._TriggeAction(tagv);
						}
					}
					else {
						if (cls.response === "sub_getsnap") {//定时读取的子站标签快照
							for (let tagv of cls.vals) {
								if (!tagv.E) {
									this._mapSnapshot.set(tagv.N, tagv);
									this._TriggeAction(tagv);
								}
							}
						}
						this._OnRecvObject(cls); //其他数据
					}
				}
			}
		};
		//安装关闭处理
		this._wsobject.onclose = (event) => {
			console.log("ws_onclose", event);
			this._loginst = false;
			for (let [key, obj] of this._mapaWait) {
				obj.reject("websocket closed."); //先逐个reject后再全部清空
			}
			this._mapaWait.clear();
			if (this.timer_checkcalltimeout) {
				clearInterval(this.timer_checkcalltimeout);
				this.timer_checkcalltimeout = 0;
			}
			if (this.timer_getsubsnaps) {
				clearInterval(this.timer_getsubsnaps);
				this.timer_getsubsnaps = 0;
			}
			this._OnWsClosed(event);
		};
		return true;
	}

	/**
	* 获取当前登录状态
	* @returns 返回bool值 true or false
	*/
	ws_islogin() {
		return this._loginst;
	}

	/**
	 * 自动登录,从sessionStorage中读取登录参数登录，用于页面切换后自动登录。
	 * @returns true or false
	 */
	ws_autologin() //自动登录
	{
		if (!this._wsobject) {
			try {
				let wsurl = sessionStorage.getItem(this._ssid_wsurl);
				let usr = sessionStorage.getItem(this._ssid_wsuser);
				let psw = sessionStorage.getItem(this._ssid_wspswd);
				if (!wsurl || !usr || !psw)
					return false;
				return this.ws_connect(wsurl, usr, psw);
			} catch (e) {
				console.log("ws_connect Err:", e);
				return false;
			}
		}
		return true;
	}

	/**
	 * 主动断开，从sessionStorage去除登录参数登录，用于页面切换后重新登录
	 * @returns 无
	 */
	ws_disconnet()//主动断开
	{
		try {
			sessionStorage.removeItem(this._ssid_wsurl);
			sessionStorage.removeItem(this._ssid_wsuser);
			sessionStorage.removeItem(this._ssid_wspswd);
			if (this._wsobject) {
				this._wsobject.close();
				this._wsobject = null;
			}
			this._loginst = false;
			this._OnLoginStatus(false);
		}
		catch (e) {
			console.log("ws_disconnet Err:", e);
		}
	}

	/**
	 * 发送消息
	 * @param {string} jstr 发送的JSON字符串
	 * @returns true or false
	 */
	ws_send(jstr)//发送消息
	{
		try {
			if (this._wsobject === null || this._wsobject.readyState !== this._wsobject.OPEN)
				return false;
			this._wsobject.send(jstr);
		} catch (e) {
			return false;
		}
		return true;
	}

	/**
	 * 从标签名中取出子站名，即':'前的字符串
	 * @param {any} tagname
	 * @returns 返回string或者null
	 */
	static tagStation(tagname) {
		var i;
		for (i = 0; i < tagname.length; i++) {
			if (tagname.charAt(i) === ':') {
				return tagname.slice(0, i);
			}
			else if (tagname.charAt(i) === '.')
				break;
		}
		return null;
	}

	/**
	 * 初始化需要订阅或者从子站定时更新的标签,追加模式，只能调用一次
	 * @param {string array} atags 标签名数组
	 */
	initTags(tags) {
		for (let i = 0; i < tags.length; i++) {
			let station = RdbView.tagStation(tags[i]);
			if (!station) {
				this._ssctags.push(tags[i]);//本站的加入订阅表
				continue;
			}
			var obj = this._stationTags.get(station);
			if (!obj) {
				obj = {};
				obj.request = "sub_getsnap";
				obj.seqno = 0;
				obj.station = station;
				obj.tags = [];
				obj.tags.push(tags[i]);
				this.stationTags.set(station, obj);
			}
			else {
				obj.tags.push(tags[i]);
			}
		}
	}

	/**
	 * 异步请求消息。
	 * @param {object} oreq 请求对象，请按照《rdb_data_exchange_protocol.pdf》填写，rdb_call会填写seqno
	 * @returns {object} 返回一个Promise对象，状态未知，调用者需await等待完成
	 */
	async rdb_call(oreq) {
		oreq.seqno = this.nextseqno();
		return new Promise((fresolve, freject) => {
			let executor = { resolve: fresolve, reject: freject, timestamp: Date.now() }; //timestamp用于检测超时
			try {
				let jstr = JSON.stringify(oreq);
				this._mapaWait.set(oreq.seqno, executor);
				if (!this.ws_send(jstr)) {
					throw new Error("ws_send error, not login or connect.");
				}
			} catch (e) {
				this._mapaWait.delete(oreq.seqno);
				freject(e);
			}
		});
	}

	/**
	 * 读取一个标签快照
	 * @param {string} stagname 标签名
	 * @returns {object} 返回一个Promise对象，状态未知，调用者需await等待完成
	 */
	async rdbGetTagVal(stagname) {
		let oreq = {};
		oreq.request = "rdb_getsnap";
		oreq.tags = [];
		oreq.tags.push(stagname);
		return this.rdb_call(oreq);
	}

	/**
	 * 读取一个标签属性
	 * @param {string} stagname 标签名
	 * @returns {object} 返回一个Promise对象，状态未知，调用者需await等待完成
	 */
	async rdbGetTagAttr(stagname) {
		let oreq = {};
		oreq.request = "rdb_tagget";
		oreq.tags = [];
		oreq.tags.push(stagname);
		return this.rdb_call(oreq);
	}

	/**
	* 控制输出
	* @param {string} stag 标签名
	* @param {string} datatype 数据类型,可以是 "digtal", "int32", "float", "double", "int64", "string", "object"之一
	* @param {any} val 标签值
	* @return {boolean}
	*/
	async rdb_ctrlout(stag, datatype, val) //控制输出，返回输出命令seqno
	{
		var oreq = {};
		oreq.request = "rdb_writedevice";
		oreq.vals = {};
		oreq.vals.N = stag;
		oreq.vals.DT = datatype;
		oreq.vals.V = val;
		return this.rdb_call(oreq);
	}

	/**
	 * 读取子站快照
	 * @returns {number} 返回发送的消息数
	 */
	_subStationGetSnaps() {
		let nmsgs = 0;
		if (!this._loginst)
			return nmsgs;
		try {
			for (let [key, obj] of this._stationTags) {
				if (!obj.tags.length)
					continue;
				obj.seqno = this.nextseqno();
				this.ws_send(JSON.stringify(obj));
				++nmsgs;
			}
		}
		catch (e) {
			console.log("_subStationGetSnaps Exception:", e);
			return nmsgs;
		}
		return nmsgs;
	}	

	/**
	 * 初始化Svg图形元素,递归调用,添加click鼠标事件
	 * @param {object} obj svg内图形元素对象
	 * anif_:闪烁; lgd 渐变颜色; anir_旋转; rctpf_ 百分比填充; am_为流动的椭圆点，不可点击。
	 */
	_InitSvgChildElement(obj) {
		let selfext = obj.getAttribute("selfext"); //点击元素的self扩展,没有返回null
		if (selfext) {
			try {
				let extobj = JSON.parse(selfext);
				if (Array.isArray(extobj)) { //支持数组
					for (const evi of extobj) {
						if (evi.evt == "onclick") {
							obj.addEventListener("click", (evt) => { this._OnElementClicked(evt); }, false);
							return;
						}
					}
				}
				else if (extobj.evt != undefined && extobj.evt == "onclick") {//也支持单个对象
					obj.addEventListener("click", (evt) => { this._OnElementClicked(evt); }, false);
					return;
				}
			} catch (e) {
				console.log("Json.parse Exception:", e);
			}
		}
		let aid = obj.getAttribute("id");
		if (aid) {
			let i;
			for (i = 0; i < this._actions.length; i++) {
				if (this._actions[i].pid === aid || "anif_" + aid === this._actions[i].pid || "lgd" + aid === this._actions[i].pid ||
					"anir_" + aid === this._actions[i].pid || "rctpf_" + aid === this._actions[i].pid) {
					obj.addEventListener("click", (evt) => { this._OnElementClicked(evt); }, false);
					return;
				}
			}
		}
		if (obj.nodeName == "g" || obj.nodeName == "foreignObject") {
			let subobj = obj.firstElementChild;
			while (subobj) {
				this._InitSvgChildElement(subobj);
				if (subobj === obj.lastElementChild)
					break;
				subobj = subobj.nextElementSibling;
			}
		}
	}

	/**
	* 初始化 SVG图形
	*/
	InitSvgEvent() {
		let obj = document.getElementById(this._svgid);
		if (!obj)
			return;
		let subobj = obj.firstElementChild;
		while (subobj) {
			this._InitSvgChildElement(subobj);
			if (subobj === obj.lastElementChild)
				break;
			subobj = subobj.nextElementSibling;
		}
	}

	/**
	* 鼠标点击处理
	* @param {object} evt 鼠标点击事件
	*/
	_OnElementClicked(evt) {
		let selfext = evt.target.getAttribute("selfext"); //获取点击元素的selext属性值,没有返回null
		if (!selfext && evt.target.parentNode.nodeName === "g") {
			selfext = evt.target.parentNode.getAttribute("selfext");
		}
		if (selfext) {
			try {
				let extobj = JSON.parse(selfext);
				if (Array.isArray(extobj)) { //支持数组
					for (let evi of extobj) {
						if (evi.evt == "onclick") {
							if (typeof OnClickedSelfExt === "function")
								OnClickedSelfExt(evt, evi);
							return; //如果selfext里有onclick事件，则不再执行原来的OnTagClicked。
						}
					}
				}
				else if (extobj.evt != undefined && extobj.evt == "onclick") {//也支持单个对象
					if (typeof OnClickedSelfExt === "function")
						OnClickedSelfExt(evt, extobj);
					return; //如果selfext里有onclick事件，则不再执行原来的OnTagClicked。
				}
			} catch (e) {
				console.log("Json.parse Exception:", e);
			}
		}
		
		//下面兼容原来模式
		let objparent = evt.target.parentNode;
		let sid = evt.target.getAttribute("id"); //点击的当前元素ID
		let sidparent = objparent.getAttribute("id");//父元素ID
		let npos = 0, n = 0;
		let exp, stag, elementid = null;
		if (!sid && !sidparent)
			return;
		if (sid) {
			if (undefined == this._mapPidActionsIdx.get(sid)) {
				if (objparent.nodeName === "g") {
					elementid = sidparent; //没有再看是不是组合元素
				}
			}
			else {
				elementid = sid;
			}
		}
		else if (objparent.nodeName === "g") {
			elementid = sidparent; //没有再看是不是组合元素
		}
		if (!elementid)
			return;

		let idxactsid = this._mapPidActionsIdx.get(elementid);
		if (undefined == idxactsid) {
			return;
		}
		if (Array.isArray(idxactsid)) {
			for (const i of idxactsid) {
				stag = null;
				exp = this._actions[i].exp;
				npos = exp.indexOf("TV(\"");
				if (npos >= 0) {
					n = exp.length;
					stag = RdbView.tagjs(exp, npos + 4, n);
					if (stag && stag.length) {
						try {
							if (typeof OnTagClicked === "function")
								OnTagClicked(stag, elementid, evt);
						} catch (e) {
							console.log("exp err:", e);
						}
						return;
					}
				} else {
					npos = exp.indexOf("TVV(\"");
					if (npos >= 0) {
						n = exp.length;
						stag = RdbView.tagjs(exp, npos + 5, n);
						if (stag && stag.length) {
							try {
								if (typeof OnTagClicked === "function")
									OnTagClicked(stag, elementid, evt);
							} catch (e) {
								console.log("exp Err:", e);
							}
							return;
						}
					}
				}
				break;
			}
		}
		try {
			if (elementid && typeof OnTagClicked === "function")
				OnTagClicked(stag, sid, evt);
		} catch (e) {
			console.log("exp err:", e);
		}
	}

	/**
	* 处理从TV("tag1")中提取"tag1",支持'\\'转义
	* @param {string} src 源串
	* @param {int32} is 开始位置
	* @param {int32} nlen 源串长度
	* @return {string} 返回处理后的字符串
	*/
	static tagjs(src, is, nlen) {
		var sr = "";
		var pre = '\0';
		var c = '\0';
		for (let j = is; j < nlen; j++) {
			c = src.charAt(j);
			if (c === '\"')
				return sr;
			else if (c === '\\') {
				if (pre === '\\') {
					sr += '\\';
					pre = '\0';
				} else
					pre = c;
			} else {
				sr += c;
				pre = c;
			}
		}
		return sr;
	}

	/**
	 * 预处理动作，生成触发器：
	 * 扫描动作表, 以标签名为key，加入action对象索引以便根据数据触发动作。
	 * 只能调用一次，在设置好_svgid, _ssctags, _subtags, _actions后调用
	 */
	PresetActions() {
		for (let i = 0; i < this._actions.length; i++) {//动作按照图形id索引
			let sid = "";
			if (this._actions[i].pid.indexOf("anif_") === 0) {
				sid = this._actions[i].pid.substring(5);
			}
			else if (this._actions[i].pid.indexOf("anir_") === 0) {
				sid = this._actions[i].pid.substring(5);
			}
			else if (this._actions[i].pid.indexOf("rctpf_") === 0) {
				sid = this._actions[i].pid.substring(6);
			}
			else if (this._actions[i].pid.indexOf("lgd") === 0) {
				sid = this._actions[i].pid.substring(3);
			}
			else {
				sid = this._actions[i].pid;
			}
			let obj = document.getElementById(sid);
			if (obj && obj.nodeName !== 'linearGradient') {
				let v = this._mapPidActionsIdx.get(sid);
				if (undefined === v) {
					this._mapPidActionsIdx.set(sid, [i]);
				}
				else {
					v.push(i);
				}
			}
		}
		for (const tagname of this._ssctags) { //建立标签名为索引的动作map
			let actidxs = [];
			for (let i = 0; i < this._actions.length; i++) {
				let npos = 0;
				let stag = "";
				npos = this._actions[i].exp.indexOf("TV(\"");
				while (npos >= 0) {
					stag = RdbView.tagjs(this._actions[i].exp, npos + 4, this._actions[i].exp.length);
					if (stag === tagname) {
						actidxs.push(i);
						break;
					}
					npos = this._actions[i].exp.indexOf("TV(\"", npos + 4);
				}
			}
			if (actidxs.length > 0) {
				this._mapTrigger.set(tagname, actidxs);
			}
		}
		for (let i = 0; i < this._actions.length; i++) { //按照对象模式修改动作里的表达式宏
			this._actions[i].exp = this._actions[i].exp.replaceAll("TV(", "this.TV(");
			this._actions[i].exp = this._actions[i].exp.replaceAll("TVV(", "this.TVV(");
			this._actions[i].exp = this._actions[i].exp.replaceAll("LV(", "this.LV(");
			this._actions[i].exp = this._actions[i].exp.replaceAll("LVV(", "this.LVV(");
			if (-1 == this._actions[i].exp.indexOf("TV(") && -1 == this._actions[i].exp.indexOf("TVV(") &&
				- 1 == this._actions[i].exp.indexOf("LV(") && -1 == this._actions[i].exp.indexOf("LVV(")) {
				if (this._actions[i].type === "a_value")
					this._do_a_value(this._actions[i]);
				else if (this._actions[i].type === "a_flash" || this._actions[i].type === "a_motion" || this._actions[i].type === "a_rotate")
					this._do_a_animation(this._actions[i]);
				else if (this._actions[i].type === "a_hidden")
					this._do_a_hidden(this._actions[i]);
				else if (this._actions[i].type === "a_color")
					this._do_a_color(this._actions[i]);
				else if (this._actions[i].type === "a_fillcolor")
					this._do_a_fillcolor(this._actions[i]);
				else if (this._actions[i].type === "a_pointer")
					this._do_a_pointer(this._actions[i]);
				else if (this._actions[i].type === "a_fillper_rect")
					this._do_a_perfill(this._actions[i]);
			}//常量表达式的动作运行一次，后面都是数据驱动，常量表达式不会再执行
		}
	}

	/**
	 * 设置标签描述，用于tooltips显示标签描述
	 * @param {object} tag 标签属性对象
	 * @returns 无
	 */
	_SetActionDesc(tag) { 
		if (tag.des == undefined)
			return;
		let idxs = this._mapTrigger.get(tag.name);//根据标签名获取动作的索引集合
		if (undefined == idxs)
			return;
		let npos = 0;
		let stag = "";
		let dlen = 4;
		for (const i of idxs) {
			stag = "";
			try {
				dlen = 4;
				npos = this._actions[i].exp.indexOf("TV(\"");
				if (npos < 0) {
					npos = this._actions[i].exp.indexOf("TVV(\"");
					dlen = 5;
				}
				if (npos >= 0) {
					stag = RdbView.tagjs(this._actions[i].exp, npos + dlen, this._actions[i].exp.length);
					if (stag && stag.length && stag === tag.name && !this._actions[i].hastitle) {
						let obj = document.getElementById(this._actions[i].pid);
						if (obj) {
							let oTitle = document.createElementNS("http://www.w3.org/2000/svg", "title");
							oTitle.innerHTML = tag.des;
							if (obj.childNodes.length > 0)
								obj.insertBefore(oTitle, obj.childNodes[0]);
							else
								obj.appendChild(oTitle);
							this._actions[i].hastitle = 1;
						}
					}
				}
			}
			catch (e) {
				console.log("_SetActionDesc exception:", e);
			}
		}
	}
	/**
	* RGB值转换为字符串 
	* @param {number} v RGB值
	* @return {string} 返回 "#RRGGBB"
	*/
	static rgb2str(v) {
		var r = "#",
			vt;
		vt = (v & 0xff0000) >> 16;
		if (vt < 16)
			r += "0" + vt.toString(16);
		else
			r += vt.toString(16);
		vt = (v & 0xff00) >> 8;
		if (vt < 16)
			r += "0" + vt.toString(16);
		else
			r += vt.toString(16);
		vt = v & 0xff;
		if (vt < 16)
			r += "0" + vt.toString(16);
		else
			r += vt.toString(16);
		return r;
	}

	/**
	 * 
	 * @param {number} ndt 标签属性里的类型
	 * @return {string} 返回控制输出的字符串.
	 */
	static datatype2str(ndt) {
		let str = "void";
		switch (ndt) {
			case 1: str = "digital"; break;
			case 2: str = "int32"; break;
			case 3: str = "float"; break;
			case 4: str = "int64"; break;
			case 5: str = "double"; break;
			case 6: str = "string"; break;
			case 7: str = "object"; break;
		}
		return str;
	}
	/**
	* 字符串转换为RGB值
	* @param {string} s "#RRGGBB"
	* @return {number} 返回RGB值
	*/
	static str2rgb(s) {
		let i;
		let v;
		let vr = 0;

		for (i = 1; i < s.length; i++) {
			v = parseInt(s.charAt(i), 16);
			if (i > 1)
				vr <<= 4;
			vr += v;
		}
		return vr;
	}

	/**
	 * 计算渐变颜色
	 * @param {string} clrfs 起始颜色
	 * @param {string} clrls 结束颜色
	 * @param {number} ni 第几段
	 * @param {number} n 总段数
	 * @return {string} 返回"#RRGGB"格式的颜色
	 */
	static calcolor(clrfs, clrls, ni, n) {
		var clrf = RdbView.str2rgb(clrfs),
			clrl = RdbView.str2rgb(clrls);
		var b1 = clrf & 0x0000ff,
			b2 = clrl & 0x0000ff,
			g1 = (clrf & 0x00ff00) >> 8,
			g2 = (clrl & 0x00ff00) >> 8,
			r1 = (clrf & 0xff0000) >> 16,
			r2 = (clrl & 0xff0000) >> 16;
		var ndr = r2 - r1,
			ndg = g2 - g1,
			ndb = b2 - b1;
		var clr = (b1 + (ni * ndb) / n) + ((g1 + (ni * ndg) / n) << 8) + ((r1 + ni * ndr / n) << 16);
		return RdbView.rgb2str(clr);
	}

	TV(tagname) {
		return this._mapSnapshot.get(tagname).V;
	}

	TVV(tagname) {
		return this._mapSnapshot.get(tagname).V;
	}

	LV(tagname) {
		return this._mapLocalVal.get(tagname).V;
	}

	LVV(tagname) {
		return this._mapLocalVal.get(tagname).V;
	}

	_do_one_exp(sexp) {
		//return new Function("", "try {return " + sexp + "; } catch(e) { return null; } ")();
		return eval(sexp);
	}

	/**
	 * 触发动作
	 * @param {object} tagval 标签值，快照值
	 */
	_TriggeAction(tagval) {
		let idxs = this._mapTrigger.get(tagval.N);
		if (undefined === idxs) {
			return;
		}
		for (let i of idxs) {
			if (this._actions[i].type === "a_value")
				this._do_a_value(this._actions[i]);
			else if (this._actions[i].type === "a_flash" || this._actions[i].type === "a_motion" || this._actions[i].type === "a_rotate")
				this._do_a_animation(this._actions[i]);
			else if (this._actions[i].type === "a_hidden")
				this._do_a_hidden(this._actions[i]);
			else if (this._actions[i].type === "a_color")
				this._do_a_color(this._actions[i]);
			else if (this._actions[i].type === "a_fillcolor")
				this._do_a_fillcolor(this._actions[i]);
			else if (this._actions[i].type === "a_pointer")
				this._do_a_pointer(this._actions[i]);
			else if (this._actions[i].type === "a_fillper_rect")
				this._do_a_perfill(this._actions[i]);
		}
	}

	/**
	 * 处理一个标签值图形动作
	 * @param {object} cls 动作对象 {type:"a_value",pid:"tagval_10",exp:"TV(\"d0.f13.pv\")",digits:3}
	 * @returns
	 */
	_do_a_value(cls) {
		try {
			let sr;
			let obj = document.getElementById(cls.pid);
			if (!obj)
				return;
			if (obj.nodeName === "g") {
				let apt = obj.getAttribute("apt"); // tagval是一个组合图形
				if (apt === "tagval") {
					sr = this._do_one_exp(cls.exp);
					if (sr === null || sr === undefined)
						return;
					try {
						sr = sr.toFixed(cls.digits);
					} catch (e) {
						console.log("exp err:", e);
					}
					document.getElementById(cls.pid + "_text").childNodes[0].nodeValue = sr;
				}
			} else if (obj.nodeName === "text") {
				sr = this._do_one_exp(cls.exp);
				if (sr === null || sr === undefined)
					return;
				try {
					sr = sr.toFixed(cls.digits);
				} catch (e) {
					console.log("exo err:", e);
				}
				document.getElementById(cls.pid).childNodes[0].nodeValue = sr;
			}
		} catch (e) {
			console.log("do_a_value err:", e);
		}
	}

	/**
	 * 执行一个动画动作
	 * @param {object} cls 动作对象
	 * {type:"a_flash",pid:"anif_text_15",exp:"TV(\"d0.dbl05.pv\") > 60",st:0 }
	 * {type:"a_rotate",pid:"anir_subgraph_82",exp:"TV(\"d0.f17.pv\") > 0",st:0}
	 * @returns
	 */
	_do_a_animation(cls) {
		try {
			let obj = document.getElementById(cls.pid);
			if (!obj)
				return;
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			if (val) {
				if (!cls.st) {
					obj.beginElement();
					cls.st = 1;
				}
			} else {
				if (cls.st) {
					obj.endElement();
					cls.st = 0;
				}
			}
		} catch (e) {
			console.log("do_a_animation err:", e);
		}
	}

	/**
	* 处理动画 a_hidden
	* @param {object} cls {type:"a_hidden",pid:"text_15",exp:"TV(\"d0.dbl05.pv\") > 60"}
	*/
	_do_a_hidden(cls) {
		try {
			let obj = document.getElementById(cls.pid);
			if (!obj)
				return;
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			if (val)
				obj.setAttribute("visibility", "visible");
			else
				obj.setAttribute("visibility", "hidden");
		} catch (e) {
			console.log("do_a_hidden err:", e);
		}
	}

	/**
	* 处理动画 a_color
	* @param {object} cls {type:"a_color",pid:"ellipse_13",pidt:0,exp: "TV(\"d0.k00.pv\")",colors:[{v:0,c:"#00FF00"},{v:1,c:"#FF0000"}]},
	*/
	_do_a_color(cls) {
		try {
			let obj = document.getElementById(cls.pid);
			if (!obj)
				return;
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			let i, n = cls.colors.length, npos = 0;
			for (i = 0; i < n; i++) {
				npos = i;
				if (val <= cls.colors[i].v)
					break;
			}
			if (obj.nodeName === "g") {
				let apt = obj.getAttribute("apt");
				if (apt === "pipe") {
					let numline = obj.getAttribute("numline");
					let clrl = obj.getAttribute("clrl");
					let clrf = obj.getAttribute("clrf");
					let clrv = cls.colors[npos].c;
					if (clrv === clrl)
						return;
					clrl = clrv;
					let clr;
					for (i = 1; i <= numline / 2; i++) {
						clr = RdbView.calcolor(clrf, clrl, i, numline / 2);
						document.getElementById(cls.pid + "_u" + i.toString()).setAttribute("stroke", clr);
						document.getElementById(cls.pid + "_d" + i.toString()).setAttribute("stroke", clr);
					}
					obj.setAttribute("clrl", clrv);
				} else if (apt === "tagval") {
					document.getElementById(cls.pid + "_text").setAttribute("fill", cls.colors[npos].c);
				}
			} else if (obj.nodeName === "text")
				obj.setAttribute("fill", cls.colors[npos].c);
			else if (obj.nodeName === "linearGradient") {
				for (i = 0; i < obj.childElementCount; i++) {
					var vc = obj.children[i].getAttribute("offset");
					if ((obj.childElementCount === 3 && (vc === "0%" || vc === "100%")) || (obj.childElementCount === 2 && vc === "100%"))
						obj.children[i].setAttribute("stop-color", cls.colors[npos].c);
				}
			} else
				obj.setAttribute("stroke", cls.colors[npos].c);
		} catch (e) {
			console.log("do_a_color err:", e);
		}
	}

	/**
	 * 处理填充颜色 a_fillcolor
	 * @param {object} cls {type:"a_fillcolor", pid:"ellipse_14",pidt:0,exp:"TV(\"d0.k00.pv\")",colors:[{v:0,c:"#00FF00"},{v:1,c:"#FF0000"}]}
	 */
	_do_a_fillcolor(cls) {
		try {
			let obj = document.getElementById(cls.pid);
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			let i, n = cls.colors.length, npos = 0;
			for (i = 0; i < n; i++) {
				npos = i;
				if (val <= cls.colors[i].v)
					break;
			}
			if (obj.nodeName === "g") {
				let apt = obj.getAttribute("apt");
				if (apt === "pipe") {
					let clrl = obj.getAttribute("clrl");
					let clrf = obj.getAttribute("clrf");
					let clrv = cls.colors[npos].c;
					if (clrf === clrv)
						return;
					obj.setAttribute("clrf", clrv);
					clrf = clrv;
					let clr = RdbView.calcolor(clrf, clrl, 1, 3);
					document.getElementById(cls.pid + "_u1").setAttribute("stroke", clr);
					document.getElementById(cls.pid + "_d1").setAttribute("stroke", clr);
					clr = RdbView.calcolor(clrf, clrl, 2, 3);
					document.getElementById(cls.pid + "_u2").setAttribute("stroke", clr);
					document.getElementById(cls.pid + "_d2").setAttribute("stroke", clr);

					clr = RdbView.rgb2str(clrf);
					document.getElementById(cls.pid + "_0").setAttribute("stroke", clr);
				} else if (apt === "tagval") {
					document.getElementById(cls.pid + "_rect").setAttribute("stroke", cls.colors[npos].c);
					document.getElementById(cls.pid + "_rect").setAttribute("fill", cls.colors[npos].c);
				}
			} else if (obj.nodeName === "linearGradient") {
				for (i = 0; i < obj.childElementCount; i++) {
					var vc = obj.children[i].getAttribute("offset");
					if ((vc === "50%") || (obj.childElementCount === 2 && vc === "0%"))
						obj.children[i].setAttribute("stop-color", cls.colors[npos].c);
				}
			} else if (obj.nodeName === "text")
				return;
			else
				obj.setAttribute("fill", cls.colors[npos].c);
		} catch (e) {
			console.log("do_a_fillcolor err:", e);
		}
	}

	/**
	 * 处理指针
	 * @param {object} cls { type: "a_pointer", pid: "pt_001", trig: 0, zx:100,zx:120,jds:180,jde360,vals:0,vale:100,exp: "_ftv(\"dec000.f0100\")" }
	 */
	_do_a_pointer(cls) {
		try {
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			let jd = cls.jds + (val - cls.vals) * (cls.jde - cls.jds) / (cls.vale - cls.vals);
			if (jd > cls.jde)
				jd = cls.jde;
			if (jd < cls.jds)
				jd = cls.jds;
			let transform = "rotate(" + jd.toString() + "," + cls.zx.toString() + "," + cls.zy.toString() + ")";
			document.getElementById(cls.pid).setAttribute("transform", transform);
		} catch (e) {
			console.log("do_a_pointer err:", e);
		}
	}

	/**
	 * 处理百分比填充
	 * @param {object} cls {type:"a_fillper_rect", pid:"rctpf_rect_97",exp:"TV(\"d0.f29.pv\")",dir:0,max:150.00000,min:0.00000,nw:56.000,nh:8.000}
	 */
	_do_a_perfill(cls) {
		try {
			let val = this._do_one_exp(cls.exp);
			if (val === null || val === undefined)
				return;
			if (cls.dir === 1)
				document.getElementById(cls.pid).setAttribute("height", (val - cls.min) * cls.nh / (cls.max - cls.min));
			else
				document.getElementById(cls.pid).setAttribute("width", (val - cls.min) * cls.nw / (cls.max - cls.min));
		} catch (e) {
			console.log("do_a_perfill err:", e);
		}
	}
}



