/**
 * JSON解析,用纯Java实现的JOSN解析，尽量不抛出异常，最少使用的接口。仅供参考，请使用java自带的json解析。
 * @author JiangYong
 * @data: 2021-03-21
 */
package kipway;

import java.util.ArrayList;

enum NodeType {
	jnone, // 没有,表示无此节点,不是节点值空的意思。
	jstring, // 字符串
	jobj, // 对象
	jarray, // 数组
	jnumber // 数字(含布尔)
}

/*
 * JSON节点非拷贝辅助类，从作者的C++库 eclib3的ec::json::t_kv翻译过来
 */
class JsonNode {
	public int kpos; // key起始位置
	public int klen; // key长度

	public int vpos; // 值起始位置
	public int vlen; // 值长度
	public NodeType vtype;// 节点类型

	public JsonNode() {
		kpos = 0;
		klen = 0;
		vpos = 0;
		vlen = 0;
		vtype = NodeType.jstring;
	}
}

/*
 * Json解析，从作者的C++库 eclib3的ec::json翻译过来
 */
public class JsonParse {

	/*
	 * JSON解析用非拷贝utf8文本辅助类，从作者的C++库 eclib3的ec::txt翻译过来，这里做成一个内部类
	 */
	class Txt {
		public int _pos;
		public int _size;

		public Txt() {
			_pos = 0;
			_size = 0;
		}

		public Txt(int pos, int size) {
			_pos = pos;
			_size = size;
		}

		public boolean empty() {
			return 0 == _size || null == _s;
		}

		public int char0() {
			if (0 == _size || null == _s || _pos >= _s.length)
				return -1;
			return _s[_pos];
		}

		public boolean findoneof(byte[] cs, byte c) {
			for (int i = 0; i < cs.length; i++) {
				if (c == cs[i])
					return true;
			}
			return false;
		}

		public boolean skip() // skip space and point to a none space char
		{
			while (_size > 0 && (_s[_pos] == 32 || _s[_pos] == '\t' || _s[_pos] == '\r' || _s[_pos] == '\n')) {
				++_pos;
				--_size;
			}
			return _size > 0;
		}

		public boolean json_tochar(int c) // for JSON
		{
			while (_size > 0) {
				if (_s[_pos] == c && _s[_pos - 1] != '\\')
					break;
				_pos++;
				_size--;
			}
			return _size > 0;
		}

		public boolean json_tochar(byte[] cs) // for JSON
		{
			while (_size > 0) {
				if (findoneof(cs, _s[_pos]) && _s[_pos - 1] != '\\')
					break;
				++_pos;
				--_size;
			}
			return _size > 0;
		}

		public boolean json_toend(int cs, int ce) // for json; v:[0,1,2,3,4]; _str point to 0, move to ;
		{
			int nk = 1;
			int cp = 0;
			while (_size > 0 && nk != 0) {
				if (_s[_pos] == '"' && _s[_pos - 1] != '\\') {
					cp = cp == '"' ? 0 : '"';
				}
				if (_s[_pos] == cs && 0 == cp && _s[_pos - 1] != '\\')
					++nk;
				else if (_s[_pos] == ce && 0 == cp && _s[_pos - 1] != '\\')
					--nk;
				++_pos;
				--_size;
			}
			return 0 == nk;
		}

		public boolean tonext() {
			if (_size > 0 && _s[_pos] != 0) {
				--_size;
				++_pos;
				return true;
			}
			return false;
		}
	}

	public final static int MAXSIZE_JSONX_KEY = 64;// key的最大长度
	private byte[] _s; // utf8源串
	private ArrayList<JsonNode> _nodes = new ArrayList<JsonNode>(128);// 节点集合

	public static int c2low(int c) {
		if (c < 'A' || c > 'Z')
			return c;
		return c + ('a' - 'A');
	}

	/*
	 * 构造参数为utf8的byte数组，即实时库jniwsapi.Call返回的byte数组。
	 */
	public JsonParse(byte[] jstrutf8) {
		_s = jstrutf8;
	}

	/*
	 * 解析,返回true表示成功
	 */
	public boolean Parse() {
		if (_s == null)
			return false;
		Txt s = new Txt(0, _s.length);
		return from_str(s);
	}

	/*
	 * 静态方法，直接返回一个JOSN解析对象
	 * 
	 * @param jstrutf8 UTF8编码的byte数组
	 */
	public static JsonParse Parse(byte[] jstrutf8) throws Exception {
		JsonParse or = new JsonParse(jstrutf8);
		if (or.Parse())
			return or;
		throw new Exception("parse failed");
	}

	/*
	 * 静态方法，把String解析后, 直接返回一个JOSN解析对象
	 * 
	 * @param jstr JSON字符串
	 */
	public static JsonParse Parse(String jstr) throws Exception {
		try {
			JsonParse or = new JsonParse(jstr.getBytes("UTF-8"));
			if (or.Parse())
				return or;
			throw new Exception("parse failed");
		} catch (Exception e) {
			throw new Exception("jstr.getBytes() failed");
		}
	}

	/*
	 * 读取并返回节点个数
	 */
	public int getNodeSize() {
		return _nodes.size();
	}

	/*
	 * 根据节点位置索引读取key名称
	 * 
	 * @param i 节点的数组索引. 范围[0,getNodeSize())
	 * 
	 * @return 返回节点的String对象，如果超限或者发生错误会抛出Exception异常。
	 */
	public String getNodeKey(int i) throws Exception {
		if (i >= _nodes.size() || i < 0) {
			throw new Exception("out of range");
		}
		JsonNode kv = _nodes.get(i);
		try {
			return new String(_s, kv.kpos, kv.klen, "UTF-8");
		} catch (Exception e) {
			throw new Exception("copy string error");
		}
	}

	/*
	 * 根据节点位置索引读取value值
	 * 
	 * @param i 节点的数组索引. 范围[0,getNodeSize())
	 * 
	 * @return 返回节点值的String对象，如果超限或者发生错误会抛出Exception异常。
	 */
	public String getNodeVal(int i) throws Exception {
		if (i >= _nodes.size() || i < 0) {
			throw new Exception("out of range");
		}
		JsonNode kv = _nodes.get(i);
		try {
			return new String(_s, kv.vpos, kv.vlen, "UTF-8");
		} catch (Exception e) {
			throw new Exception("copy string error");
		}
	}

	/*
	 * 根据节点位置索引读取节点类型
	 * 
	 * @param i 节点的数组索引. 范围[0,getNodeSize())
	 * 
	 * @return 返回节点类型的NodeType枚举值，返回NodeType.jnone表示超限无此节点。
	 */
	public NodeType getNodeType(int i) {
		if (i >= _nodes.size() || i < 0)
			return NodeType.jnone;
		JsonNode kv = _nodes.get(i);
		return kv.vtype;
	}

	/*
	 * 不分大小写比较两个相同长度的utd8 byte[]的内容是否相同。
	 */
	public boolean ieq(byte[] s1, int pos1, byte[] s2, int pos2, int size) {
		for (int j = 0; j < size; j++) {
			if (c2low(s1[pos1 + j]) != c2low(s2[pos2 + j]))
				return false;
		}
		return true;
	}

	/*
	 * 根据key名称读取value值
	 * 
	 * @return 如果不存在，则会返回一个空字符串(不是null)，应用根据情况可以使用默认值(google protocol buffer 规则)
	 */
	public String getNodeVal(String key) {
		JsonNode kv;
		try {
			byte[] sk = key.getBytes("UTF-8");
			for (int i = 0; i < _nodes.size(); i++) {
				kv = _nodes.get(i);
				if (kv.klen == sk.length && ieq(sk, 0, _s, kv.kpos, kv.klen))
					return new String(_s, kv.vpos, kv.vlen, "UTF-8");
			}
		} catch (Exception e) {
		}
		return new String();
	}

	/*
	 * 根据key名称读取节点类型
	 * 
	 * @return 如果不存在，则会返回NodeType.jnone，应用根据情况可以使用默认值(google protocol buffer 规则)
	 */
	public NodeType getNodeType(String key) {
		JsonNode kv;
		try {
			byte[] sk = key.getBytes("UTF-8");
			for (int i = 0; i < _nodes.size(); i++) {
				kv = _nodes.get(i);
				if (kv.klen == sk.length && ieq(sk, 0, _s, kv.kpos, kv.klen))
					return kv.vtype;
			}
		} catch (Exception e) {
		}
		return NodeType.jnone;
	}

	/*
	 * 根据节点位置索引解析对象或数组节点为一个 JsonParse对象
	 * 
	 * @param i 节点的数组索引. 范围[0,getNodeSize())
	 * 
	 * @return 返回一个解析后的JsonParse对象
	 * 
	 * @remark 节点必须是NodeType.jobj或者NodeType.jarray类型，否则会抛出异常。
	 */
	public JsonParse ParseNode(int i) throws Exception {
		if (i >= _nodes.size() || i < 0) {
			throw new Exception("out of range");
		}
		JsonNode kv = _nodes.get(i);
		if (kv.vtype != NodeType.jarray && kv.vtype != NodeType.jobj) {
			throw new Exception("node is not a array or object");
		}
		JsonParse jp = new JsonParse(_s);
		if (!jp.Parse(kv.vpos, kv.vlen))
			throw new Exception("Parse failed");
		return jp;
	}

	/*
	 * 根据节点名解析对象或数组节点为一个 JsonParse对象
	 * 
	 * @param i 节点的数组索引. 范围[0,getNodeSize())
	 * 
	 * @return 返回一个解析后的JsonParse对象
	 * 
	 * @remark 节点必须是NodeType.jobj或者NodeType.jarray类型，否则会抛出异常。
	 */
	public JsonParse ParseNode(String key) throws Exception {
		JsonNode kv;
		try {
			byte[] sk = key.getBytes("UTF-8");
			for (int i = 0; i < _nodes.size(); i++) {
				kv = _nodes.get(i);
				if (kv.klen == sk.length && ieq(sk, 0, _s, kv.kpos, kv.klen)) {
					if (kv.vtype != NodeType.jarray && kv.vtype != NodeType.jobj) {
						throw new Exception("node is not a array or object");
					}
					JsonParse jp = new JsonParse(_s);
					if (!jp.Parse(kv.vpos, kv.vlen))
						throw new Exception("Parse failed");
					return jp;
				}
			}
		} catch (Exception e) {
			throw new Exception("key.getBytes failed");
		}
		throw new Exception("not find node '" + key + "'");
	}

	/*
	 * 打印空格，用于格式化输出
	 */
	public static void prtspace(int n) {
		for (int i = 0; i < n; i++)
			System.out.print(" ");
	}

	/*
	 * 递归解析和打印JSON节点，可以参考这个函数的实现解析JSON消息。
	 * 
	 * @param js JsonParse对象
	 * 
	 * @param nspace 打印开始的缩进空格数
	 * 
	 * @param tabsize 制表符空格个数
	 */
	public static boolean printJson(JsonParse js, int nspace, int tabsize) {
		NodeType kvtype;
		String key, val;
		try {
			for (int i = 0; i < js.getNodeSize(); i++) {
				key = js.getNodeKey(i);
				kvtype = js.getNodeType(i);
				prtspace(nspace);
				if (kvtype == NodeType.jarray) {
					if (key.isEmpty())
						System.out.print("[\n");
					else
						System.out.print("\'" + key + "\":[\n");
					JsonParse jsnode = js.ParseNode(i);
					printJson(jsnode, nspace + tabsize, tabsize);
					prtspace(nspace);
					System.out.println("]" + ((i + 1 != js.getNodeSize()) ? ',' : '\0'));
				} else if (kvtype == NodeType.jobj) {
					if (key.isEmpty())
						System.out.print("{\n");
					else
						System.out.print("\"" + key + "\":{\n");
					JsonParse jsnode = js.ParseNode(i);
					printJson(jsnode, nspace + tabsize, tabsize);
					prtspace(nspace);
					System.out.println("}" + ((i + 1 != js.getNodeSize()) ? ',' : '\0'));
				} else if (kvtype == NodeType.jstring) {
					val = js.getNodeVal(i);
					System.out.println(
							"\"" + key + "\": " + "\"" + val + "\"" + ((i + 1 != js.getNodeSize()) ? ',' : '\0'));
				} else {
					val = js.getNodeVal(i);
					System.out.println("\"" + key + "\": " + val + ((i + 1 != js.getNodeSize()) ? ',' : '\0'));
				}

			}
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	private boolean Parse(int pos, int size) {
		if (_s == null)
			return false;
		Txt s = new Txt(pos, size);
		return from_str(s);
	}

	private boolean from_str(Txt s) {
		_nodes.clear();
		if (s.empty())
			return false;
		if (!s.skip())
			return false;
		if (s.char0() == '[')
			return from_array(s);
		else if (s.char0() == '{')
			return from_obj(s);
		return false;
	}

	private boolean from_obj(Txt s) {
		if (s.char0() != '{')
			return false;
		s.tonext();
		byte[] cend = { ',', '}' };
		while (!s.empty()) {
			if (!s.skip()) // to key start
				return false;
			if (s.char0() == ',') {
				s.tonext();
				continue;
			} else if (s.char0() == '}')
				return true;
			if (s.char0() != '"')
				return false;
			JsonNode it = new JsonNode();
			s.tonext();
			it.kpos = s._pos; // key
			if (!s.json_tochar('"')) // to key end
				return false;
			it.klen = s._pos - it.kpos;
			if (0 == it.klen || it.klen > MAXSIZE_JSONX_KEY) // check key
				return false;
			if (!s.json_tochar(':')) // move to valuse
				return false;
			s.tonext();
			if (!s.skip())
				return false;
			switch (s.char0()) {
			case '"':
				s.tonext();
				it.vtype = NodeType.jstring;
				it.vpos = s._pos;
				if (!s.json_tochar('"'))
					return false;
				it.vlen = s._pos - it.vpos;
				s.tonext();
				break;
			case '{':// object
				it.vpos = s._pos;
				it.vtype = NodeType.jobj;
				s.tonext();
				if (!s.json_toend('{', '}'))
					return false;
				it.vlen = s._pos - it.vpos;
				break;
			case '[':// array
				it.vpos = s._pos;
				it.vtype = NodeType.jarray;
				s.tonext();
				if (!s.json_toend('[', ']'))
					return false;
				it.vlen = s._pos - it.vpos;
				break;
			default: // Number and boolean
				it.vpos = s._pos;
				it.vtype = NodeType.jnumber;
				if (!s.json_tochar(cend))
					return false;
				it.vlen = s._pos - it.vpos;
				break;
			}
			_nodes.add(it);
		}
		return false;
	}

	private boolean from_array(Txt s) {
		if (s.char0() != '[')
			return false;
		byte[] cend = { ',', '}' };
		s.tonext();
		JsonNode it;
		while (!s.empty()) {
			if (!s.skip())
				return false;
			switch (s.char0()) {
			case ']': // end
				return true;
			case ',':
				s.tonext();
				break;
			case '"': // string
				s.tonext();
				it = new JsonNode();
				it.vtype = NodeType.jstring;
				it.vpos = s._pos;
				if (!s.json_tochar('"'))
					return false;
				it.vlen = s._pos - it.vpos;
				_nodes.add(it);
				s.tonext();
				break;
			case '{':// object
				it = new JsonNode();
				it.vpos = s._pos;
				it.vtype = NodeType.jobj;
				s.tonext();
				if (!s.json_toend('{', '}'))
					return false;
				it.vlen = s._pos - it.vpos;
				_nodes.add(it);
				break;
			case '[':// array
				it = new JsonNode();
				it.vpos = s._pos;
				it.vtype = NodeType.jarray;
				s.tonext();
				if (!s.json_toend('[', ']'))
					return false;
				it.vlen = s._pos - it.vpos;
				_nodes.add(it);
				break;
			default: // Number and boolean
				it = new JsonNode();
				it.vpos = s._pos;
				it.vtype = NodeType.jnumber;
				if (!s.json_tochar(cend))
					return false;
				it.vlen = s._pos - it.vpos;
				_nodes.add(it);
				break;
			}
		}
		return false;
	}
}
