﻿/*!
\file rdbapi.h
\brief 实时库接口API定义 for windows and linux

\update 
  2024-9-19  增加订阅推送接口和相关错误码定义,GBK和UTF8互转函数,ISO8601时标转换函数。
  2023-12-20 修改rdb_valquery接口lflag参数定义,增加功能
  2023-3-31  增加数据删除、增强插入、工作模式接口
	rdb_datadelete
	rdb_soedelete
	rdb_valinsertex
	rdb_objinsertex
	rdb_getworkmode
	rdb_setworkmode
  2023-2-19 增加获取当前实时库的内部版本接口rdb_curdb_inver
  2021-9-18 增加标签扩展属性接口。
  2021-3-1  监控接口和订阅子站SOE接口。
  2020-12-6 增加混合读取标签快照接口rdb_getsnapshot。
  2020-8-17 增加切换式双库连接。
  2020-6-14 增加人工置数接口。
  2020-3-22 使用基于WS的协议实现,兼容前面应用。
  2008-8-23 增加SOE订阅
*/

#ifndef RDBAPI_H
#define RDBAPI_H

#include "rdbdef.h"

#define DBR_SUCCESS         0       //成功
#define SE_ERRHANDLE        (-1)    //错误的句柄

//错误码
#define SE_OK				0x00	//成功
#define SE_SUCCESS			SE_OK	//成功
#define SE_FAILED			0x01	//失败
#define SE_TCP_CONNECT		0x02    //TCP连接错误
#define SE_TCP_PROXY_CONNECT 0x03   //SOCKET5代理错误
#define SE_TCP_PROXY_AUTH	0x04    //SOCKET5代理验证错误
#define SE_TCP_DISCONNECT	0x05    //TCP连接断开
#define SE_TCP_IO			0x06    //TCP网络IO错误
#define SE_TCP_CONSVRFAIL	0x07    //TCP连接服务器失败
#define SE_TCP_SENDBLOCK	0x08    //TCP发送阻塞
#define SE_TCP_ERRPKG		0x0A    //TCP报文错误
#define SE_TCPCALLTIMEOUT	0x0B    //TCP调用超时
#define SE_ERRARGS          0x0C    //调用参数错误

#define SE_EXP				0x10	//表达式错误
#define SE_EXPVAR			0x11	//表达式引用变量错误
#define SE_EXPVAROVER		0x12	//表达式引用变量太多，最多允许4个
#define SE_BUSY				0x13	//服务器忙，未完成请求任务
#define SE_VPFDATAPAGE      0x14	//虚拟页面文件中数据页面错误
#define SE_TIME             0x15	//时标错误
#define SE_ARCHIVE          0x16	//归档错误
#define SE_NOTCONNECT		0x1F	//没有连接

#define SE_OPT_OK			0		//成功
#define SE_OPT_NOACTOR		0x20	//无此角色
#define SE_OPT_NOUSER		0x21	//无此账号
#define SE_OPT_PASSERR		0x22	//密码错误
#define SE_OPT_NOPOWER		0x23	//无此权限
#define SE_OPT_USEREXIST	0x24	//账号重复
#define SE_OPT_ACTOREXIST	0x25	//角色重复
#define SE_OPT_ACTORUSE		0x26	//角色正在使用
#define SE_OPT_NOLOGIN		0x27	//没有登录
#define SE_OPT_NOTACTIVE	0x28	//账号冻结
#define SE_OPT_OPTFULL		0x29	//账号满
#define SE_OPT_ACTFULL		0x2A	//角色满
#define OPT_NOTMODPSW		0x2B	//不容许修改密码

#define SE_TG_NOTAG			0x30	//无此标签
#define SE_TG_TAGFULL		0x31	//标签满
#define SE_TG_NOARCHIVE		0x32	//不归档
#define SE_TG_ERRNAME		0x33	//标签名错误
#define SE_TG_NOSTATION		0x34	//无子站或者子站不在线

#define SE_DA_TIME			0x40	//时标错误
#define SE_DA_TYPE			0x41	//类型错误
#define SE_DA_QA			0x42	//质量错误
#define SE_DA_NODATA		0x43	//无数据
#define SE_DA_END			0x44	//数据结束，这个不是错误，是用于查询记录结束用的
#define SE_DA_SIZEOVER		0x48	//数据长度超限
#define SE_DA_INSERT		0x49	//数据插入错误

#define SE_SSC_OVER_TAGS	0x4A	//匹配订阅超限
#define SE_SSC_OVER_SESSIONS 0x4B	//订阅会话数超限
#define SE_SSC_NOTSSC		0x4C	//未订阅,取消订阅未订阅的标签时返回这个状态码

#define SE_CTRLNOROUTE		0x61	//无路由不能写
#define SE_CTRLOUTERR       0x62    //输出错误

#define SE_SRV_FRMNOOD		0xFF	//无此命令
#define SE_SRV_FRMERR		0xFE	//报文错误
#define SE_SRV_DBVER		0xFD	//API和实时库版本不一致
#define SE_SRV_BACKUOPSRV   0XFB    //备份服务器
#define SE_SRV_DEMOOVER		0xFC	//试用过期,历史数据服务停止

///\brief标签查询回调函数
///return false表示终止查询
typedef bool(*rdbcb_OnReadTags)(rec_tag ptags[], int nitems, void* pParam);

///\brief值记录查询回调函数
///return false表示终止查询
typedef bool(*rdbcb_OnReadVals)(rec_val pvals[], int nitems, void* pParam);

///\brief对象记录查询回调函数
///return false表示终止查询
typedef bool(*rdbcb_OnReadObjs)(rec_obj pobjs[], int nitems, void* pParam);

///\brief SOE记录查询回调函数
///return false表示终止查询
typedef bool(*rdbcb_OnReadSoes)(rec_soe psoes[], int nitems, void* pParam);

///\brief 服务器推送的订阅SOE记录回调函数
typedef void(*rdbcb_OnSscPutSoes)(rec_soe psoes[], int nitems, void* param);

///\brief 值标签写设备回调函数
///\return 0表示失败,非零表示成功
typedef bool(*ctrl_OnValWriteToDevice)(rec_tagval* pval, void* pParam);

///\brief 对象标签写设备回调函数
///\return 0表示失败,非零表示成功
typedef bool(*ctrl_OnObjWriteToDevice)(rec_tagobj* pobj, void* pParam);

///\breif 服务器推送的订阅标签快照值数据回调函数
typedef void(*rdbcb_OnPushValSnaps)(const rec_tagval pvals[], int sizeVals, void* param);

///\breif 服务器推送的订阅标签快照对象数据回调函数
typedef void(*rdbcb_OnPushObjSnaps)(const rec_tagobj pobjs[], int sizeObjs, void* param);

///\breif 服务器推送的JSON消息(包括SOE和订阅数据推送)回调函数,适合高级用户自己解析订阅的SOE和订阅的快照数据JSON报文
typedef void(*rdbcb_OnPushJsonMsgs)(const char* jstr, int sizejstr, void* param);

#define MSGCODE_CONNECT_SUCCESS  60001   //!<异步连接成功
#define MSGCODE_CONNECT_FAILED   60002   //!<异步连接失败
#define MSGCODE_DISCONNECTED     60003   //!<连接断开,包含接收数据和发送数据时

/*!
\brief 消息通知回调函数
\param nmsgcode,MSGCODE_CONNECT_SUCCESS,MSGCODE_CONNECT_FAILED,MSGCODE_DISCONNECTED之一
*/
typedef void(*rdbcd_onmessage)(int nmsgcode, const char* smsg, void* param);

#ifdef __cplusplus
extern "C" {
#endif

	/*!
	\brief创建接口
	\return 返回句柄,-1表示失败，>=0为成功创建的句柄号
	\remark 句柄为大于等于0的整数,一个句柄代表一个API对象，如果要建立多个到实时库的连接,可以创建多个句柄，每个进程最多可以创建64个句柄。

	\code {.cpp}
	int h = rdb_create();
	if(h == SE_ERRHANDLE)
		printf("error\n");
	\endcode
	*/
	int  rdb_create();
	
	/*!
	\brief 获取当前连接的实时库
	\return 0:主库; 1:从库; -1表示失败	
	\remark fileversion 5.0.6.9开始有效
	*/
	int  rdb_curdb(int h);
	
	/*!
	\brief 设置从库
	\param h [in] rdb_create创建的句柄.
	\param sip [in] ipv4地址.
	\param wport [in] 服务端口.
	\param sUser [in] 账号.
	\param sPass [in] 账号密码.
	\return 0:成功; -1表示失败
	\remark fileversion 5.0.6.9开始有效, 在rdb_create之后，connect之前设置从库的参数，从库的协议和主库相同，
	   即主库是ws协议则从库也是ws协议通道，主库是wss协议则从库也是wss协议通道。
	   主库和从库采用切换机制，同一时刻只有一个活动连接，主库失败或断开后尝试使用从库参数连接，从库失败或断开后尝试使用主库参数连接。
	   只有断开才会发生切换连接操作。	   
	*/
	int  rdb_setslavedb(int h, const char* sip, unsigned short wport, const char* sUser, const char* sPass);
	
	/*!
	\brief删除句柄
	\param h [in] rdb_create创建的句柄.
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 不再使用实时库连接后,使用这个函数删除句柄,释放系统资源,使用前先调用rdb_disconnect断开连接.
	*/
	int  rdb_destory(int h);

	/*!
	\brief 高级连接,rdb2020.3版开始增加的接口
	\param surl 连接URL字符串, "wss://192.168.161:921" 或者 "ws://192.168.161:921",或者"wss://kipwa.net"
	\param sUser 账号
	\param sPass 密码
	\param nTimeOutSec 0表示异步连接，立即返回，大于0表示同步方式，等待超时秒数
	*/
	int rdb_connectex(int h, const char* surl, const char* sUser, const char* sPass, int nTimeOutSec);

	/*!
	\brief连接实时库
	\param h [in] rdb_create创建的句柄.
	\param sServerIP [in] 实时库服务器IP地址,点分字符串格式,比如"192.168.1.61".
	\param wPort [in] 实时库TCP服务端口.
	\param lpszUser [in] 用户账号名.
	\param lpszPass [in] 用户账号密码.
	\return 返回 0 表示成功，其余值为失败错误码
	*/
	int  rdb_connect(int    h, const char *sServerIP, unsigned short wPort, const char * lpszUser, const char * lpszPass);

	/*!
	\brief异步连接实时库
	\param h [in] rdb_create创建的句柄.
	\param sServerIP [in] 实时库服务器IP地址,点分字符串格式,比如"192.168.1.61".
	\param wPort [in] 实时库TCP服务端口.
	\param lpszUser [in] 用户账号名.
	\param lpszPass [in] 用户账号密码.
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 异步连接实时库，结果通过rdbcd_onmessage回调通知
	*/
	int  rdb_connectasyn(int    h, const char *sServerIP, unsigned short wPort, const char * lpszUser, const char * lpszPass);

	/*!
	\brief断开实时库
	\param h [in] rdb_create创建的句柄.
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 不再使用实时库连接后,先使用rdb_disconnect断开连接,然后调用rdb_destory销毁句柄释放资源.
	*/
	int  rdb_disconnect(int h);

	/*!
	\brief判断连接状态
	\param h [in] rdb_create创建的句柄.
	\return 返回 非0 表示处于连接状态; 0表示处于断开状态
	*/
	bool rdb_isconnected(int h);

	/*!
	\brief添加用户角色
	\param h [in] rdb_create创建的句柄.
	\param pa [in] 角色结构体指针
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int  rdb_actoradd(int h, rec_actor* pa);

	/*!
	\brief删除用户角色
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 角色名
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int	 rdb_actordel(int h, const char *sname);

	/*!
	\brief读取角色表
	\param h [in] rdb_create创建的句柄.
	\param actors [out] 用于接收角色记录的结构体数组
	\param nrecbufsize [in] actors数组大小,能容纳角色记录的个数
	\param pnacts [out] 实际读取到并存放到actors数组中的记录数
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int	 rdb_actorsget(int h, rec_actor actors[], int nrecbufsize, int *pnacts);

	/*!
	\brief添加用户账号
	\param h [in] rdb_create创建的句柄.
	\param po [in] 账号结构体指针
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int	 rdb_operatoradd(int h, rec_operator* po);

	/*!
	\brief删除用户账号
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 账号名
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int	 rdb_operatordel(int h, const char *sname);

	/*!
	\brief读取账号表
	\param h [in] rdb_create创建的句柄.
	\param opts [out] 用于接收账号记录的结构体数组
	\param nrecbufsize [in] opts数组大小,能容纳账号记录的个数
	\param pnopts [out] 实际读取到并存放到opts数组中的记录数
	\return 返回 0 表示成功，其余值为失败错误码
	\remark 需要账号管理权限 OPT_POW_MAN
	*/
	int	 rdb_operatorsget(int h, rec_operator opts[], int nrecbufsize, int *pnopts);

	/*!
	\brief更改自己密码
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 自己的用户名
	\param soldpass [in] 旧密码
	\param snewpass [in] 新密码
	\return 返回 0 表示成功，其余值为失败错误码
	*/
	int	 rdb_operatormodifypass(int h, const char *sname, const char*soldpass, const char *snewpass);

	/*!
	\brief导入(添加,更改)标签
	\param h [in] rdb_create创建的句柄.
	\param dwflag [in] 导入策略,指明当标签存在时修改哪些字段,参见TAG_MASK_DES
	\param ptag [in] 标签记录指针
	\param presult [out] 处理结果; 0未更改; 1修改了; 2添加了
	\return 返回 0 表示成功，其余值为失败错误码
	*/
	int  rdb_taginport(int h, unsigned int dwflag, rec_tag* ptag, int  *presult);

	/*!
	\brief读取标签属性记录
	\param h [in] rdb_create创建的句柄.
	\param ptag [in/out] 标签属性记录,调用前填写sname字段.
	\return 返回 0 表示成功,其余值为失败错误码。成功调用后,ptag的其余字段被填写.
	*/
	int	 rdb_tagget(int h, rec_tag* ptag);

	/*!
	\brief删除标签
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 要删除的标签名.
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 删除标签的同时删除历史数据并释放实时库内部逻辑存储页面。
	*/
	int  rdb_tagdel(int h, const char *sname);

	/*!
	\brief查询标签
	\param h [in] rdb_create创建的句柄.
	\param snamefilter [in] 标签名过滤串,空或*表示全部,支持*和?匹配
	\param sdesfilter [in] 标签描述过滤串,空或*表示全部,支持*和?匹配
	\param ndatatype [in] 标签名数据类型,-1表示全部
	\param ntagclass [in] 标签分类,-1表示全部
	\param pfun [in] 回调函数,参见rdbcb_OnReadTags,查询到标签记录后使用回调函数处理.
	\param pParam [in] 回调函数的参数.
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 当查询结束或者回调函数返回0时结束查询，也就是调用者可以中途终止查询.整个查询过程中，回调函数可能被调用多次.
	*/
	int  rdb_tagquery(int h, const char* snamefilter, const char* sdesfilter, int  ndatatype, int  ntagclass, rdbcb_OnReadTags pfun, void* pParam);

	/*!
	\brief写值标签快照(写值标签实时数据)
	\param h [in] rdb_create创建的句柄.
	\param vals [in/out] 标签值记录数组,rdb_valputsnapshot返回0时,每个记录中的cerr为0表示该记录成功,否则为错误码
	\param nsize [in] vals中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 写快照是追加模式,生产现场的实时数据使用这个接口写入到实时库,实时库会根据例外和压缩参数处理。这个接口提交数据记录时，服务器的处理能力大于100万记录每秒
	从rdb1911开始(rdbsrv file version 5.0.6.0)，如果时标填写0则表示快照时标使用实时库服务器的当前时间作为时标，特别适合对于无实时钟或者时钟无法校时的数据采集网关提交快照数据。
	注意对于DEC(设备)标签，新数据的时标必须大于旧数据的时标。对于PRESET标签，新数据的时标大于等于旧数据的时标。
	*/
	int  rdb_valputsnapshot(int h, rec_tagval vals[], int nsize);

	/*!
	\brief读取值标签快照(读值标签实时数据)
	\param h [in] rdb_create创建的句柄.
	\param vals [in/out] 标签值记录数组,调用前填写每个记录结构体中的标签名sname字段
	\param nsize [in] vals中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。调用成功后,还需判断每个记录的vals[i].val.cerr错误码,cerr为0表示有效,否则可能是标签不存在或者没有数据.
	*/
	int  rdb_valgetsnapshot(int h, rec_tagval vals[], int nsize);

	/*!
	\brief 补录值标签数据(写值标签历史数据)
	\param h [in] rdb_create创建的句柄.
	\param vals [in] 标签值记录数组
	\param nsize [in] vals中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 补录是插入模式,如果前置机到服务器的网络通道故障，可以将数据暂存在前置机本地,故障恢复后可以补录未提交的数据,补录和最新的快照(实时数据)可以同时向服务器提交。
	补录属于历史插入,也支持压缩,但不支持例外,补录的的效率主要受服务器存储IO性能的影响(大约在1千到10万记录每秒)。实际应用中可单独开启一个补录连接使用单独线程处理补录数据。
	*/
	int  rdb_valinsert(int h, rec_tagval vals[], int nsize);

	/*!
	\brief读取值标签历史断面数据
	\param h [in] rdb_create创建的句柄.
	\param ltime [in] 断面时标,见T_TIME定义
	\param vals [in/out] 标签值记录数组,调用前填写每个记录结构体中的标签名sname字段
	\param nflag [in] 断面处理标识,0:插值; -1:时标小于等于ltime的样本值记录; 1:时标大于等于ltime的样本值记录;
	\param nsize [in] vals中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。调用成功后,还需判断每个记录的vals[i].val.cerr错误码,cerr为0表示有效,否则可能是标签不存在或者没有数据.
	\remark 此函数PRESET标签不反回数据。
	*/
	int  rdb_valgetsection(int h, T_TIME ltime, int nflag, rec_tagval vals[], int nsize);

	/*!
	\brief 查询值标签历史数据记录
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 标签名
	\param sexp [in] 查询表达式
	\param lts [in] 开始时标,参见T_TIME定义
	\param lte [in] 结束时标,参见T_TIME定义,-1表示直到读完
	\param lds [in] 插值间隔时间,单位为100毫秒数, 0表示读取样本值
	\param lflag [in] 读取标识<br>
		0: 默认模式，QA = 1的shutdown的时间段数据不插值。<br>
		1: 绘图数据，插值包含样本值，因此插值出来的记录数据多余预计的记录数，QA = 1的shutdown的时间段数据不插值<br>
		2: 全数据插值，不管数据质量<br>
		3: 全数据绘图数据，不管数据质量，插值包含样本值，因此插值出来的记录数据多余预计的记录数<br>
		4: 读取区间最小值，内部版本5114及以后支持。<br>
		5: 读取区间最大值，内部版本5114及以后支持。<br>
		6: 读取区间最小值，不管数据质量，内部版本5114及以后支持。<br>
		7: 读取区间最大值，不管数据质量，内部版本5114及以后支持。<br>
		注1）4，5，6，7模式dt 必须 > 0表示区间(等于0会返回样本值)，表达式sexp参数被忽略。当区间内无样本值记录时，会采用插值方式补齐，插值模式根据标签属性中的定义决定使用梯形还是线性插值。4，5模式QA = 1的shutdown的时间段数据不插值；6，7模式则不关心QA的值。<br>
		注2）1，3模式绘图数据从内部版本5114开始，表达式sexp参数被忽略；lds为0表示服务器通过time_begin和time_end值自动计算间隔；

	\param pfun [in] 数据记录到达回掉函数,参见rdbcb_OnReadVals
	\param pParam [in] pfun使用的参数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 直到数据查询完成或者回调函数返回0表示结束查询,调用者可以随时通过回调函数返回0结束查询。整个查询过程中，回调函数可能被调用多次.
	*/
	int	 rdb_valquery(int h, const char *sname, const char *sexp, T_TIME lts, T_TIME lte, T_TIME lds, int  lflag, rdbcb_OnReadVals pfun, void* pParam);

	/*!
	\brief写对象标签快照(写对象标签实时数据)
	\param h [in] rdb_create创建的句柄.
	\param pobjs [in/out] 对象标签记录数据数组
	\param nsize [in] pobjs中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码,调用成功后,还需判断每个记录的pobjs[i].var.cerr错误码。
	\remark 写快照是追加模式,生产现场的DT_STRING和DT_OBJECT型实时数据使用这个接口写入到实时库。
	从rdb1911开始(rdbsrv file version 5.0.6.0)，如果时标填写0则表示快照时标使用实时库服务器的当前时间作为时标，特别适合对于无实时钟或者时钟无法校时的数据采集网关提交快照数据。
	注意对于DEC(设备)标签，新数据的时标必须大于旧数据的时标。对于PRESET标签，新数据的时标大于等于旧数据的时标。
	*/
	int	 rdb_objputsnapshot(int h, rec_tagobj pobjs[], int nsize);

	/*!
	\brief读取对象标签快照数据(读对象标签实时数据)
	\param h [in] rdb_create创建的句柄.
	\param pobjs [in/out] 对象标签数据记录数组,调用前填写每个记录结构体中的标签名sname字段
	\param nsize [in] pobjs中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。调用成功后,还需判断每个记录的pobjs[i].var.cerr错误码,cerr为0表示有效,否则可能是标签不存在或者没有数据.
	*/
	int  rdb_objgetsnapshot(int h, rec_tagobj pobjs[], int nsize);

	/*!
	\brief 补录对象标签历史数据(写对象标签历史数据)
	\param h [in] rdb_create创建的句柄.
	\param pobjs [in] 对象标签数据记录数组
	\param nsize [in] pobjs中的记录数.
	\return 返回 0 表示成功,其余值为失败错误码。
	*/
	int  rdb_objinsert(int h, rec_tagobj pobjs[], int nsize);

	/*!
	\brief 读取对象标签历史数据
	\param h [in] rdb_create创建的句柄.
	\param stagname [in] 标签名
	\param lts [in] 开始时标,参见T_TIME
	\param lte [in] 结束时标,参见T_TIME，-1表示直到读取完
	\param pfun [in] 数据到达回调函数，参见rdbcb_OnReadObjs
	\param pParam [in] 回调函数的参数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 直到数据查询完成或者回调函数返回0表示结束查询,调用者可以随时通过回调函数返回0结束查询。整个查询过程中，回调函数可能被调用多次.
	*/
	int  rdb_objget(int h, const char* stagname, T_TIME lts, T_TIME lte, rdbcb_OnReadObjs pfun, void* pParam);

	/*!
	\brief 写入SOE事件
	\param h [in] rdb_create创建的句柄.
	\param psoe [in] SOE事件记录数组
	\param nsize psoe中的记录个数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 这个函数为写入新的SOE事件，每个SOE记录的uautokey字段填写0,追加和历史补录均使用本函数;
	*/
	int  rdb_soeput(int h, rec_soe psoe[], int nsize);

	/*!
	\brief 查询SOE事件
	\param h [in] rdb_create创建的句柄.
	\param lts [in] 开始时标,参见T_TIME
	\param autokey [in] 开始时标处的autokey
	\param lte [in] 结束时标,参见T_TIME,-1表示查询到数据结束
	\param sexp [in] 查询表达式
	\param pfun [in] 查询结果记录回调函数
	\param pParam [in]回调参数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 直到查询完成或者回调函数返回0表示结束查询,调用者可以随时通过回调函数返回0结束查询。整个查询过程中，回调函数可能被调用多次.
	*/
	int  rdb_soequery(int h, T_TIME lts, unsigned int autokey, T_TIME lte, const char * sexp, rdbcb_OnReadSoes pfun, void* pParam);

	/*!
	\brief 更新SOE事件记录
	\param h [in] rdb_create创建的句柄.
	\param psoe [in/out] 事件记录数组,返回时cstatus字段会被改写
	\param nsize [in] psoe中的记录数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark SOE事件使用time字段和uautokey字段组合主键来决定唯一的SOE记录,当返回0后，还需判断每个SOE记录中的status字段为0表示成功，1表示更新失败
	*/
	int  rdb_soeupdate(int h, rec_soe psoe[], int nsize);

	/*!
	\brief 订阅SOE事件
	\param h [in] rdb_create创建的句柄.
	\param flag [in] 订阅模式; 0:取消订阅; 1:订阅最新; 2:订阅更新的SOE(被处理后的SOE); 3:订阅最新和更新后的SOE.
	\return 返回 0 表示成功,其余值为失败错误码。
	*/
	int	 rdb_soesubscription(int h, int flag);

	/*!
	\brief 设置处理订阅的SOE事件处理回调函数
	\param h [in] rdb_create创建的句柄.
	\param fun [in] 回调函数
	\param param [in] 回调函数自己带入的参数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 在调用rdb_soesubscription前调用本函数设置处理订阅推送回来的SOE事件
	*/
	int  rdb_soesubscription_setfun(int h, rdbcb_OnSscPutSoes fun, void* param);

	/*!
	\brief 值标签数据记录统计
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 标签名
	\param sexp [in] 表达式
	\param lts [in] 开始时标,参见T_TIME
	\param lte [in] 结束时标,参见T_TIME
	\param pMin [out] 最小值出现处的记录
	\param pMax [out] 最大值出现处的记录
	\param pavg [out] 平均值
	\param psum [out] 累计值
	\param plrecs [out]  记录数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 此函数PRESET标签不反回数据。
	*/
	int  rdb_countvalue(int  h, const char* sname, const char* sexp, T_TIME  lts, T_TIME  lte, rec_val* pMin, rec_val* pMax, double *pavg, double *psum, int  *plrecs);

	/*!
	\brief 值标签状态改变统计
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 标签名
	\param lts [in] 开始时标,参见T_TIME
	\param lte [in] 结束时标,参见T_TIME
	\param nlowval [in]  低电平值或者断开状态值
	\param nhighval [in] 高电平值或者闭合状态值
	\param pnl2h [out] 断开到闭合的改变次数
	\param pnh2l [out] 闭合到断开的改变次数
	\return 返回 0 表示成功,其余值为失败错误码。
    \remark 常用于统计开关或断路器动作次数，此函数PRESET标签不反回数据。
	*/
	int  rbd_countstatuschang(int h, const char* sname, T_TIME  lts, T_TIME lte, int nlowval, int nhighval, int *pnl2h, int *pnh2l);

	/*!
	\brief 值标签数据c持续时间统计
	\param h [in] rdb_create创建的句柄.
	\param sname [in] 标签名
	\param sexp [in] 表达式
	\param lts [in] 开始时标,参见T_TIME
	\param lte [in] 结束时标,参见T_TIME
	\param pltime [out] 满足表达式条件的数据持续时间
	\param plrecs [out] 满足表达式条件的数据样本值记录数
	\return 返回 0 表示成功,其余值为失败错误码。
    \remark 常用于统计数据合格率，此函数PRESET标签不反回数据。
	*/
	int  rdb_countvaltime(int h, const char* sname, const char *sexp, T_TIME lts, T_TIME lte, T_TIME *pltime, int *plrecs);

	/*!
	\brief 获取当前时间的实时库时标
	\return 返回当前时间的T_TIME值
	*/
	long long  rdb_time();

	/*!
	\brief 本地时间合成为实时库时标T_TIME
	\param nyear [in] 年1970-...
	\param nmon  [in] 月1-12
	\param nday  [in] 日1-31
	\param nhour [in] 时0-23
	\param nmin  [in] 分0-59
	\param nsec  [in] 秒0-59
	\param nmsec [in] 毫秒0-999
	\return 返回实时库时标T_TIME
	*/
	long long  rdb_localtime2rdbtime(int nyear, int nmon, int nday, int nhour, int nmin, int nsec, int nmsec);

	/*!
	\brief 实时库时标T_TIME分解为本地时间
	\param ltime  [in] 实时库时标
	\param pnyear [out]
	\param pnmon  [out]
	\param pnday  [out]
	\param pnhour [out]
	\param pnmin  [out]
	\param pnsec  [out]
	\param pnmsec [out]
	\return 返回true表示成功，false表示失败
	*/
	bool  rdb_rdbtime2localtime(T_TIME ltime, int *pnyear, int *pnmon, int *pnday, int *pnhour, int *pnmin, int *pnsec, int *pnmsec);

	/*!
	\brief 初始化写设备回调函数
	\param h [in] rdb_create创建的句柄.
	\param funvalw [in] 值标签写设备回调函数
	\param paramvalw [in] 值标签写设备回调函数的pParam参数
	\param funobjw [in] 对象标签写设备回调函数
	\param paramobjw [in] 对象标签写设备回调函数的pParam参数
	\return 0表示成功,其余值为失败错误码。
	\remark 只有数据采集和控制前置机需要将实时库标签写入到现场设备时才需要设置写设备回调函数。
	*/
	int  rdb_initwritetodevice(int h, ctrl_OnValWriteToDevice funvalw, void* paramvalw, ctrl_OnObjWriteToDevice funobjw, void* paramobjw);

	/*!
	\brief 注册控制标签
	\param h [in] rdb_create创建的句柄.
	\param stagname [in] 标签名
	\return 0表示成功,其余值为失败错误码。
	\remark 只有数据采集和控制前置机需要将实时库标签写入到现场设备时才需要使用本函数。这个函数的目的是向实时库报告stagname由当前连接负责写入现场设备，相当于建立一条
	由实时库写入现场设备的路由。当前连接断开后，实时库自动清除stagname的写设备路由信息。
	*/
	int  rdb_regctrltag(int h, const char* stagname);

	/*!
	\brief  写值标签的值到现场设备
	\param h [in] rdb_create创建的句柄.
	\param pval [in] 值标签数据记录
	\return 0表示成功,其余值为失败错误码。
	\remark 写入现场设备的标签必须是通过rdb_regctrltag由自己或其他连接注册了标签,已经建立了写设备路由，否则会返回SE_CTRLNOROUTE错误。
	*/
	int  rdb_writevaltodevice(int h, rec_tagval *pval);

	/*!
	\brief  写对象标签的值到现场设备
	\param h [in] rdb_create创建的句柄.
	\param pobj [in] 对象标签数据记录
	\return 0表示成功,其余值为失败错误码。
	\remark 写入现场设备的标签必须是通过rdb_regctrltag由自己或其他连接注册了标签,已经建立了写设备路由,否则会返回SE_CTRLNOROUTE错误。
	*/
	int  rdb_writeobjtodevice(int h, rec_tagobj *pobj);

	/*!
	\brief 设置消息回掉函数
	\param h [in] rdb_create创建的句柄.
	\param fun [in] 处理消息通知的回调函数,注意和调用者不在一个线程中
	\param param [in] 回调函数自己使用的参数
	*/
	int  rdb_setmessagenotify(int h, rdbcd_onmessage fun, void* param);

	/*!
	\brief 实时库时间(从1970-1-1开始的100毫秒数)转换为字符串
	\param ltime实时库时间
	\param sout 输出空间
	\param outsize 输出空间大小(字节数)
	\return 返回输出的字符数,意义同snprintf德返回值,如果错误返回-1; 如果返回值大于等于outsize则输出被截断。
	*/
	int rdbtime_tostr(int64_t ltime, char *sout, size_t outsize);

	/*!
	\brief 本地字符串表达的时间转换为实时库时间
	\param str 字符串，比如"2019/12/3 12:24:56.200" 或者  "2019/12/3 12:24:56"
	\param zlen  str的长度(字符数)
	\return 返回实时库时间(从1970-1-1开始的100毫秒数)
	*/
	int64_t rdbtime_fromstr(const char *str, size_t zlen);

	/*!
	\brief 计算md5散列值
	\param pdata 需要编码的数据
	\param nsize  pdata的长度(字符数)
	\param out  输出区,最小16字节
	\return 无，结果16字节在out中。
	*/
	void  rdb_md5_encode(const void* pdata, int nsize, unsigned char out[16]);

	/*!
	\brief base64快速编码
	\param pdata 需要编码的数据
	\param nsize  pdata的长度(字符数)
	\param sout  输出区
	\param noutsize  输出区长度(字符数)
	\return 返回字节数,输出带0结尾，空间不够返回-1; 当sout为nullptr时,返回需要的输出空间大小
	*/
	int  rdb_base64_encode(const void* pdata, int nsize, char *sout, int noutsize);

	/*!
	\brief base64快速解码
	\param pdata 需要解码的数据
	\param nsize  pdata的长度(字符数)
	\param sout  输出区
	\param noutsize  输出区长度(字符数)
	\return 返回字节数,输出带0结尾，空间不够返回-1; 当sout为nullptr时,返回需要的输出空间大小
	*/
	int  rdb_base64_decode(const char* pdata, int nsize, unsigned char *sout, int noutsize);

	/*!
	\brief 根据错误码获取错误描述
	\param nerrcode 错误码
	\return 返回错误信息。
	*/
	const char* rdb_geterrmsg(int nerrcode);

	/*!
	\brief 设置人工置数
	\param h[in] rdb_create创建的句柄.
	\param pval [in] 标签和预设值的数据
	\return 返回错误码，0表示成功。
	*/
	int  rdb_manual_set(int h, rec_tagval *pval);

	/*!
	\brief 取消人工置数
	\param h[in] rdb_create创建的句柄.
	\param stagname[in] 标签名
	\return 0表示成功, 其余值为失败错误码。
	*/
	int  rdb_manual_del(int h, const char* stagname);

	/*!
	\brief 读取当前越限标签快照
	\param h[in] rdb_create创建的句柄.
	\param station[in] 站名,nullptr表示主站
	\param onread[in] 数据回调函数
	\param puserparam回调函数的参数,会原样填写到onread的param里。
	\return 0表示成功, 其余值为失败错误码。
	*/
	int	rdb_getoutlimit_snaps(
		int h,
		const char *station,
		void(*onread)(rec_tagval pvals[], int nitems, void* param),
		void* puserparam);
		
	/*!
	\brief 读取快照,不分值标签还是对象标签
	\param h[in] rdb_create创建的句柄.
	\param tagnames[in] 标签名字符串数组
	\param nsize [in] 标签个数
	\param funval [in] 值数据回调函数
	\pram  param_val [in] funval的最后一个参数。
	\param funobj [in] 对像数据回调函数
	\pram  param_obj [in] funobj的最后一个参数。
	\return 0表示成功, 其余值为失败错误码。
	\remark 每个回调函数可能被多次回调. rdbapi fileversion 5.0.7.2新增
	*/
	int rdb_getsnapshot(
		int h,//in 句柄
		const char* tagnames[],	//标签名数组
		int nsize,//个数
		void(*funval)(const rec_tagval vals[], size_t valsize, void *param), void* param_val,
		void(*funobj)(const rec_tagobj objs[], size_t objsize, void *param), void* param_obj
	);
	
	
	/*!
	\brief 向实时库请求消息; 
	\param h[in] rdb_create创建的句柄.
	\param request [in] 请求的消息，JSON字符串，utf8编码
	\param requestsize [in] request的长度，字节数
	\param fun [in] 接收消息回调函数。
	\param app_param [in] 应用层传递的参数, fun函数的最后一个参数.
	\return 0: success; 其他为错误码
	\remark 返回消息通过回调函数返回，response为JSON应答字符串,sizeresponse为应答字符串的长度(字节数)。这个函数可以用于调用以后增加的消息。
	*/
	int rdb_callmsg(int h, const char* request, int requestsize, void(*fun)(const char* response, int sizeresponse, void* app_param), void* app_param);
	
	/*!
	\brief 向实时库推送消息
	\param h[in] rdb_create创建的句柄.
	\param jstr [in] 推送的JSON消息，utf8编码
	\param jstrsize [in] jstr的长度，字节数
	\return 0: success; 其他为错误码
	\remark 这是内部用的，目前用于rdbdac_ux定时推送网关运行信息.
	*/
	int rdb_putmsg(int h, const char* jstr, int jstrsize);
	
	/*!
	\brief 订阅SOE的高级版，支持通过中心站订阅子站SOE
	\param h[in] rdb_create创建的句柄.
	\param flag[in] 订阅模式; 0:取消订阅; 1:订阅最新; 2:订阅更新的SOE(被处理后的SOE); 3:订阅最新和更新后的SOE.
	\param station[in] 子站名，null表示订阅目前连接的站的SOE(和rdb_soesubscription功能同)。
	\return 0: success; 其他为错误码
	*/
	int	 rdb_soesubscription_ex(int h, int flag, const char* station);
	
	/*!
	\brief 查询SOE事件高级版，支持通过中心站从子站查询
	\param h [in] rdb_create创建的句柄.
	\param station[in] 子站名，null表示从目前连接的站查询(和rdb_soequery功能同)。
	\param lts [in] 开始时标,参见T_TIME	
	\param autokey [in] 开始时标处的autokey
	\param lte [in] 结束时标,参见T_TIME,-1表示查询到数据结束
	\param sexp [in] 查询表达式
	\param pfun [in] 查询结果记录回调函数
	\param pParam [in]回调参数
	\return 返回 0 表示成功,其余值为失败错误码。
	\remark 直到查询完成或者回调函数返回0表示结束查询,调用者可以随时通过回调函数返回0结束查询。整个查询过程中，回调函数可能被调用多次.
	*/
	int  rdb_soequery_ex(int h, const char* station
		, T_TIME  lts, unsigned int autokey, T_TIME lte, const char * sexp
		, rdbcb_OnReadSoes pfun, void* puserparam);

	/*!
	\brief 从中心站查询当前在线子站列表和概要信息
	\param h [in] rdb_create创建的句柄.
	\param fun [in] 接收消息回调函数。
	\param app_param [in] 应用层传递的参数, fun函数的最后一个参数.
	\return 0: success; 其他为错误码
	\remark 返回消息通过回调函数返回，jsresp为JSON应答字符串,sizeresp为应答字符串的长度(字节数)。返回的JSON对象定义参见协议文档《rdb_data_exchange_protocol.pdf》8.1。
	*/
	int rdb_liststation(int h, void(*fun)(const char* jsresp, int sizeresp, void* app_param), void *app_param);
	
	/*!
	\brief 从中心站查询子站运行信息
	\param h [in] rdb_create创建的句柄.
	\param stationid[in] 子站ID，null表示从目前连接的站查询。stationid用 "子站名.master" 表示子站主站，"子站id.slave" 表示子站从站。
	\param fun [in] 接收消息回调函数。
	\param app_param [in] 应用层传递的参数, fun函数的最后一个参数.
	\return 0: success; 其他为错误码
	\remark 返回消息通过回调函数返回，jsresp为JSON应答字符串,sizeresp为应答字符串的长度(字节数)。返回的JSON对象定义参见协议文档《rdb_data_exchange_protocol.pdf》8.2。
	*/
	int rdb_getruninfo(int h, const char* stationid, void(*fun)(const char* jsresp, int sizeresp, void* app_param), void *app_param);
	
	/*!
	\brief 导入标签扩展属性
	\param h [in] rdb_create创建的句柄.
	\param pext[in] 扩展属性结构体。
	\param umask [in] 掩码
	\return 0: success; 其他为错误码
	*/
	int rdb_tagext_inport(int h, tag_ext* pext, uint32_t umask);//导入单个标签的扩展属性.
	
	/*!
	\brief 获取一批标签扩展属性
	\param h [in] rdb_create创建的句柄.
	\param ptags[in] in/out 标签扩展属性数组,输入时只填写标签名
	\param size [in] ptags中标签个数
	\return 0: success; 其他为错误码
	*/
	int rdb_tagext_get(int h, tag_ext* ptags, int size);
	
	/*!
	\brief 删除一个标签的扩展属性
	\param h [in] rdb_create创建的句柄.
	\param tagname[in] 标签名
	\return 0: success; 其他为错误码
	*/
	int rdb_tagext_del(int h, const char* tagname);
	
	/*!
	\brief 获取当前实时库的内部版本
	\param h [in] rdb_create创建的句柄.
	\return -1:失败; >0 内部版本号，比如5104
	*/
	int rdb_curdb_inver(int h);

	/**
	 * @brief 删除标签数据记录, 双机热备时，只删除当前连接的实时库中数据。
	 * @param tagname 标签名
	 * @param datetime_begin 开始日期时间(含)
	 * @param datetime_end 结束日期时间(不含)，nullptr或者空字符串表示一直删除到结束
	 * @param pnumdelrecords 成功后，回填删除的记录数。
	 * @return 返回0成功，其他为错误码
	*/
	int rdb_datadelete(int h, const char* tagname, const char* datetime_begin, const char* datetime_end, int* pnumdelrecords);

	/**
	 * @brief 删除SOE记录, 双机热备时，只删除当前连接的实时库中数据。
	 * @param datetime_begin 开始日期时间,ISO格式"2023-1-1T08:0:0.000+08:00"
	 * @param ukey_begin 开始序号
	 * @param datetime_end 结束日期时间(ISO格式), nullptr或者空字符串表示一直删除到结束
	 * @param ukey_end 结束序号
	 * @param pnumdelrecords 成功后输出删除的记录数
	 * @return 返回0成功，其他为错误码
	*/
	int rdb_soedelete(int h, const char* datetime_begin, unsigned int ukey_begin, const char* datetime_end, unsigned int ukey_end, int* pnumdelrecords);

	/**
	 * @brief 值类型标签数据记录增强插入,无压缩插入一个标签的多条记录,用于备份或者数据维护,如果存在，不相同更改。 双机热备时，只在当前连接的实时库中插入数据。
	 * @param h rdb_create创建返回的句柄
	 * @param tagname 标签名
	 * @param vals 记录
	 * @param numvals 记录数
	 * @param pninsert 回填的插入记录数
	 * @param pnupdate 回填的更新记录数据
	 * @param pnignore 回填的忽略记录数,完全相同无更改
	 * @return 返回0成功，其他为错误码
	*/
	int rdb_valinsertex(int h, const char* tagname, rec_val vals[], int numvals, int* pninsert, int* pnupdate, int* pnignore);

	/**
	 * @brief 对像类型标签数据记录增强插入一个标签的多条记录,用于备份或者数据维护,如果存在，不相同更改。双机热备时，只在当前连接的实时库中插入数据。
	 * @param h rdb_create创建返回的句柄
	 * @param tagname 标签名
	 * @param objs 记录
	 * @param numobjs 记录数
	 * @param pninsert 回填的插入记录数
	 * @param pnupdate 回填的更新记录数据
	 * @param pnignore 回填的忽略记录数,完全相同无更改
	 * @return 返回0成功，其他为错误码
	*/
	int rdb_objinsertex(int h, const char* tagname, rec_obj objs[], int numobjs, int* pninsert, int* pnupdate, int* pnignore);

	/**
	* @brief 读取工作模式
	* @param h rdb_create创建的句柄
	* @param pworkmode 成功会回填工作模式； 0：work模式； 1：管理模式
	* @return 成功返回0，否则返回错误码.
	* @remark 实时库单独部署时，work模式和manage模式完全一样，没有区别；
		   双机热备部署时，在默认的工作模式下，以下命令对备机的请求会转发到主机处理：
		   rdb_getsnap
		   rdb_getoutlimitsnaps
		   rdb_valgetsection
		   rdb_objget
		   rdb_valquery
		   rdb_plotdata
		   rdb_sum
		   rdb_countkst
		   rdb_countruntime
		   rdb_soequery

		   设置为manage模式就可以强制不转发，只对连接的服务器操作。可以避免误会，比如已经使用rdb_datadelete命令删除了备机上的数据，结果在work模式还能查到数据。
	*/
	int rdb_getworkmode(int h, int* pworkmode);

	/**
	* @brief 设置工作模式
	* @param h rdb_create创建的句柄
	* @param workmode 0：work模式； 1：管理模式
	* @param pworkmode 成功会回填工作模式； 0：work模式； 1：管理模式
	* @return 成功返回0，否则返回错误码.
	* @remark 实时库单独部署时，work模式和manage模式完全一样，没有区别；
		   双机热备部署时，在默认的工作模式下，以下命令对备机的请求会转发到主机处理：
		   rdb_getsnap
		   rdb_getoutlimitsnaps
		   rdb_valgetsection
		   rdb_objget
		   rdb_valquery
		   rdb_plotdata
		   rdb_sum
		   rdb_countkst
		   rdb_countruntime
		   rdb_soequery

		   设置为manage模式就可以强制不转发，只对连接的服务器操作。
	*/
	int rdb_setworkmode(int h, int workmode, int* pworkmode);

	/**
	 * @brief 设置处理实时库推送json消息的处理回调函数，用于高级用户自己解析订阅推送的快照和SOE
	 * @param h rdb_create创建的句柄
	 * @param fun 处理接收json消息的回调函数
	 * @param pfunparam 回调函数自己的参数
	 * @return 返回0成功，其他为错误码
	 * @remark rdb_create成功后就设置,或者登录成功后设置,也可以在在调用rdb_sscsnaps前设置,推荐rdb_create成功后设置。设置后回调函数包括其pfunparam必须
	 * 保持有效可用。注意这个回调函数是在rdbapi后台运行时线程被调用，注意pfunparam参数多线程数据访问冲突。适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_pushjsonmessage_setfun(int h, rdbcb_OnPushJsonMsgs fun, void* pfunparam);

	/**
	 * @brief 设置订阅标签快照处理回调函数
	 * @param h rdb_create创建的句柄
	 * @param funVal 标签值数据回调函数
	 * @param pValparam 值数据回调函数的param参数，回调时原样压缩参数栈。
	 * @param funObj 标签对象数据回调函数
	 * @param pObjparam 对象数据回调函数的param参数，回调时原样压缩参数栈。
	 * @return 返回0成功，其他为错误码
	 * @remark rdb_create成功后就设置,或者登录成功后设置,也可以在在调用rdb_sscsnaps前设置,推荐rdb_create成功后设置。设置后回调函数包括其param必须
	 * 保持有效可用。注意这个回调函数是在rdbapi后台运行时线程被调用，注意param参数多线程数据访问冲突。适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_subscription_setfun(int h, rdbcb_OnPushValSnaps funVal, void* pValparam, rdbcb_OnPushObjSnaps funObj, void* pObjparam);

	/**
	 * @brief 订阅快照,订阅后，当有新的快照数据写入实时库时，对每个订阅者推送一份快照数据。
	 * @param h rdb_create创建的句柄
	 * @param mode 订阅模式；0:订阅后不推送当前快照;1:订阅需要推送当前快照; 当前快照是指订阅前就存在的快照,这个mode不影响订阅后新快照的推送。
	 * @param codepage 指示参数字符串和回调函数参数字符串使用的编码；0:GBK, 1:utf8
	 * @param tagnames 需要订阅的标签名字符串数组,字符串编码由codepage指定,标签名可以最多有16个通配标签("d0.*","d1.*"等，"*"表示全部)和不限个数的精确指定标签。
	 * @param nsize 标签个数
	 * @param funErr 错误回调，有多少个错误标签回调多少次。
	 * @param paramFun 错误回调函数的param参数。
	 * @return 返回0成功，其他为错误码
	 * @remark 订阅快照保存在服务器的连接会话了，连接断开后订阅表在服务端被删除，因此每次登录成功后都需要使用这个函数提交订阅表。适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_sscsnaps(
		int h,  //in 句柄
		int mode, //0:订阅后不推送随后一个记录;1:订阅需要推送最后一笔记录
		int codepage, //0:GBK, 1:utf8, 指示参数字符串和回调函数参数字符串使用的编码
		const char* tagnames[],//标签名数组
		int nsize,	//个数
		void(*funErr)(const char* tagname, int errcode, void* param),//错误标签回调，有多错误回调多少次。
		void* paramFun //funErr里面的param
	);

	/**
	 * @brief 取消订阅快照
	 * @param h rdb_create创建的句柄
	 * @param codepage 指示参数字符串和回调函数参数字符串使用的编码；0:GBK, 1:utf8
	 * @param tagnames 需要取消订阅的标签名数组,字符串编码由codepage指定
	 * @param nsize 标签个数
	 * @param funErr 错误回调,有多少个错误标签回调多少次。
	 * @param paramFun 错误回调的param参数。
	 * @return 返回0成功，其他为错误码
	 * @remark 适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_unsscsnaps(
		int h,  //in 句柄
		int codepage, //0:GBK, 1:utf8, 指示参数字符串和回调函数参数字符串使用的编码
		const char* tagnames[],//标签名数组
		int nsize,	//个数
		void(*funErr)(const char* tagname, int errcode, void* param),//错误标签回调，有多错误回调多少次。
		void* paramFun //funErr里面的param
	);

	/**
	 * @brief 取消所有订阅
	 * @param h rdb_create创建的句柄
	 * @return 返回0成功，其他为错误码
	 * @remark 适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_unsscallsnaps(
		int h  //in 句柄
	);

	/**
	 * @brief 读取当前自己的订阅表
	 * @param h rdb_create创建的句柄
	 * @param codepage 指示参数字符串和回调函数参数字符串使用的编码；0:GBK, 1:utf8
	 * @param funTag 标签回调,有多少标签回调多少次。
	 * @param paramFun 回调函数funTag的param参数
	 * @return 返回0成功，其他为错误码
	 * @remark 适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_listsscsnaps(
		int h,  //in 句柄
		int codepage, //0:GBK, 1:utf8, 指示参数字符串和回调函数参数字符串使用的编码
		void(*funTag)(const char* tagname, void* param),//订阅标签回调，有多少标签回调多少次。
		void* paramFun //funTag里面的param
	);//读取订阅表

	/**
	 * @brief rdbapi的文件版本
	 * @return 返回本api动态库自身的文件版本，整数，例如： 5101
	 * @remark 适合rdb2024.9 inver=5119及以后版本。
	 */
	int rdb_apiver();

	/**
	* @brief GBK转UTF8
	* @param sgbk 输入的GBK
	* @param gbksize 输入的GBK长度(字节数)
	* @param sout 输出区，成功输出后会在最末尾添加0
	* @param outsize 输出区长度
	* @return 返回转换后的字节数(不含结束符0),-1表示失败。
	*/
	int rdb_gbk2utf8(const char* sgbk, size_t gbksize, char* sout, size_t outsize);

	/**
	 * @brief UTF8转GBK
	 * @param sutf8 输入的UTF8字符串
	 * @param utf8size 输入的UTF8长度(字节数)
	 * @param sout 输出区，成功输出后会在最末尾添加0
	 * @param outsize 输出区长度
	 * @return 返回转换后的字节数(不含结束符0),-1表示失败。
	 */
	int rdb_utf82gbk(const char* sutf8, size_t utf8size, char* sout, size_t outsize);

	/**
	 * @brief js时标转ISO8601格式字字符串时标,例"2023-2-20T12:10:32.123+08:00"
	 * @param jtime, int64, 自1970-1-1开始的毫秒数,UTC时标(0时区)
	 * @param out 输出的字符串缓冲区，
	 * @param outsize 输出字符串缓冲区大小，建议大于等于32字节;
	 * @param useLocalTimeZone 输出时使用本地时区；0：不使用； 1：使用本地时区
	 * @return 转换的字节数
	 * @remark useLocalTimeZone=1 输出"2024-9-20T12:10:32.123+08:00"; =0输出 "2029-9-20T4:10:32.123Z"
	 */
	int rdb_jtime2string(long long jtime, char* out, size_t outsize, int useLocalTimeZone);

	/**
	 * @brief ISO8601字符串格式的时标转换为jtime(int64, 自1970-1-1开始的毫秒数,UTC时标(0时区))
	 * @param sdatetime ISO8601字符串格式时标或者本地时标格式。
	 * @param zlen sdatetime的长度
	 * @return int64, -1:时标，大于等于0成功,  自1970-1-1开始的毫秒数,UTC时标(0时区)
	 * @remark sdatetime 支持以下格式：
		ISO 8601
			2023-2-31T04:32:25.987Z
			2023-2-31T12:32:25.987+0800
			2023-2-31T12:32:25.987+08:00
			2023-2-31T12:32:25+08:00
			2023-2-31T12:32+08:00
			2023-2-31T12+08:00
			2023-2-31Z
		local datetime
			2023-1-31 12:32:25.987
			2023-1-31 12:32:25
			2023-1-31 12:32
			2023-1-31 12
			2023-1-31
	 */
	long long rdb_string2jtime(const char* sdatetime, size_t zlen);

#ifdef __cplusplus
}
#endif

#endif //RDBAPI_H
