﻿/*!
\file pagetable.js
分页表格

\author jiangyong
\update 
	2025-4-21 V1.0.1 对象实现改为函数模式加原型模式
	2022-7-17 V1.0.0
*/

/**
 * 分页表格对象
 * @param {string} objvarname 对象实例名
 * @param {function} makePageHtml 页面html生成函数
 * @param {string} idtabbody 表格body的id
 * @param {string} idpagination 分页导航的id
 * @param {string} idpaginationdiv 分页导航所在div的id，用于显示隐藏
 * @param {number} numpagerow //每页行数
 * @param {string} idtxtrownums //显示总行数的标签id，label,p等
 */
function PageTable(objvarname, makePageHtml, idtabbody, idpagination, idpaginationdiv, numpagerow = 20, idtxtrownums = "") {
	this.pagerow = numpagerow;//每页行数
	this.pageno = 0;//当前画面第一页
	this.recs = new Array();//记录集对象数组
	this.pagenum = 0; //总页数
	this.pagepage = 10; //每次左右导航翻页页面数
	this.pageview = 0; //当前显示页
	this.idtabbody = idtabbody;//表格体ID
	this.idtxtnums = idtxtrownums;//显示总行数数的文本lable标签元素
	this.idpagination = idpagination;//分页标签的ID
	this.idpaginationdiv = idpaginationdiv;//分页标签所在div的ID,用于显示隐藏
	this.objvarname = objvarname; //用于标签响应脚本里
	this.makePageHtml = makePageHtml; //生成pageview页面的html，由外部实现	
	this.afterViewPage = null; //显示页面后立即调用,用于先显示标签在显示标签值的场景.
}

PageTable.prototype = {
	constructor: PageTable,

	/**
	* 设置记录数组,并显示翻页控件
	* @param {object array} recs
	*/
	setRecords: function (recs) {
		this.recs = recs;
		if (this.idtxtnums != "") {
			$('#' + this.idtxtnums).text("Num:" + recs.length);
		}
		this.pagenum = parseInt(this.recs.length / this.pagerow);
		if ((this.recs.length % this.pagerow) != 0)
			this.pagenum++;
		if (this.idpaginationdiv != "") {
			document.getElementById(this.idpaginationdiv).style.visibility = (this.pagenum > 1) ? "visible" : "hidden";
		}
	},

	/**
	* 清空分页表格
	*/
	clearPageTable: function () {
		this.recs.length = 0;
		this.pagenum = 0;
		this.pageno = 0;
		this.pageview = 0;
		if (this.idpaginationdiv != "") { //隐藏分页导航
			document.getElementById(this.idpaginationdiv).style.visibility = "hidden";
		}
		if (this.idtxtnums != "") { //清除总页数显示
			$('#' + this.idtxtnums).text("Num:");
		}
		$('#' + this.idpagination).html("");//清除分页导航条
		$('#' + this.idtabbody).html("");//清除表格内容    
	},

	/**
	* 设置行数，重新显示
	* @param {number} n
	*/
	setPageRowNum: function (n) {
		if (n == this.pagerow)
			return;
		let curphr = this.pagerow * this.pageview;
		this.pagerow = n;
		if (this.pagerow < 10)
			this.pagerow = 10;
		else if (this.pagerow > 1000)
			this.pagerow = 1000;
		this.pagenum = parseInt(this.recs.length / this.pagerow);
		if ((this.recs.length % this.pagerow) != 0)
			this.pagenum++;
		if (this.idpaginationdiv != "") {
			if (this.pagenum > 1) {
				document.getElementById(this.idpaginationdiv).style.visibility = (this.pagenum > 1) ? "visible" : "hidden";
			}
		}
		this.viewPage(parseInt(curphr / this.pagerow));
	},

	/**
	* 后翻(左)pagepape页
	*/
	onPaginationLeft: function () {
		if (this.pageno == 0)
			return;
		this.pageno -= this.pagepage;
		if (this.pageno < 0)
			this.pageno = 0;
		this.viewPage(this.pageno);
	},

	/**
	 * 前翻(右)pagerow页
	 */
	onPaginationRight: function () {
		if (this.pageno + this.pagepage >= this.pagenum)
			return;
		this.pageno += this.pagepage;
		this.viewPage(this.pageno);
	},

	/**
	 * 显示翻页导航条
	 */
	viewPagination: function () {
		let ps = this.pageno, i = 0;
		var shtml = "<li> <a  href='#' onclick = '" + this.objvarname + ".onPaginationLeft();'> <span aria-hidden=\"true\">&laquo;</span> </a></li>";
		while (ps < this.pagenum && i < this.pagepage) {
			if (ps == this.pageview)
				shtml += "<li class='active'><a href='#' aria-hidden='true' onclick='" + this.objvarname + ".viewPage(" + ps + ");'> <span aria-hidden=\"true\">" + ps + "</span></a></li>";
			else
				shtml += "<li><a href='#' aria-hidden='true' onclick='" + this.objvarname + ".viewPage(" + ps + ");'> <span aria-hidden=\"true\">" + ps + "</span></a></li>";
			i++;
			ps++;
		}
		shtml += "<li> <a href='#' onclick = '" + this.objvarname + ".onPaginationRight();'> <span aria-hidden=\"true\">&raquo;</span> </a></li>";
		$('#' + this.idpagination).html(shtml);
	},

	/**
	 * 显示表格数据
	 * @param {number} npg 页号
	 */
	viewPage: function (npg) {
		this.pageview = npg;
		if (this.pageview >= this.pagenum) {
			this.pageview >= this.pagenum - 1;
		}
		else if (this.pageview < 0)
			this.pageview = 0;
		document.getElementById(this.idtabbody).innerHTML = this.makePageHtml();
		this.pageno = this.pagepage * parseInt(npg / this.pagepage);
		this.viewPagination();

		if (this.afterViewPage)
			this.afterViewPage();
	},
}

